from collections import defaultdict

from utils import utils

YUID_ALL_COLUMNS = ['yuid', 'ua_profile', 'ip_activity_type']

def prepare_device_watch_log(rec):
    yield {'devid': rec['key'],
           'ip': rec['ip'],
           'timestamp': rec['timestamp'],
           'rec_type': 'watch',
           'ua_profile': rec['ua_profile']}


def join_yuid_and_rlogin_for_logged_in_yuids(ip_key, recs):
    login = ''
    login_yuids = defaultdict(set)
    for rec in recs:
        rec_type = rec['rec_type']
        if rec_type == 'watch':
            yuid = rec['yuid']
            if yuid != '0' and login:  # fetch only already logged in yuids
                login_yuids[login].add(yuid)
        else:
            login = rec['login']

    for login, yuids in login_yuids.iteritems():
        for yuid in yuids:
            yield {'yuid': yuid, 'login': login}


def join_logs_by_ip_and_ts(ip_key, recs):
    login = ''
    for rec in recs:
        rec_type = rec['rec_type']
        if rec_type == 'watch':
            yuid = rec['yuid']
            if yuid != '0' and login:  # fetch only already logged in yuids
                yield {'ip': ip_key['ip'], 'timestamp': rec['timestamp'], 'yuid': yuid, 'login': login}
        else:
            login = rec['login']


def join_device_and_rlogin_for_logged_in_yuids(ip_key, recs):
    login = ''
    login_devices = defaultdict(set)
    for rec in recs:
        rec_type = rec['rec_type']
        if rec_type == 'watch':
            devid = rec['devid']
            ua_profile = rec['ua_profile']
            if login:  # fetch only already logged in yuids
                login_devices[login].add((devid, ua_profile))
        else:
            login = rec['login']

    for login, devices in login_devices.iteritems():
        for (devid, ua_profile) in devices:
            yield {'login': login, 'devid': devid, 'ua_profile': ua_profile}


def remove_login_conflicts_by_first(yuid_key, recs):
    # Assume first hit is the best
    first_rlogin = None
    first_rlogin_ts = None
    for r in recs:
        login = r['login']
        if not first_rlogin:
            first_rlogin = login
            first_rlogin_ts = r['timestamp']
        if first_rlogin == login:
            r['@table_index'] = 0
        else:
            # yuid can't have several logins
            r['first_login'] = first_rlogin
            r['first_login_dt_ts'] = utils.ts_to_datetime_str(first_rlogin_ts)
            r['first_login_dt'] = utils.ts_to_date_str(first_rlogin_ts)

            ts = r['timestamp']
            r['dt_ts'] = utils.ts_to_datetime_str(ts)
            r['dt'] = utils.ts_to_date_str(ts)

            r['@table_index'] = 1
        yield r


def remove_login_conflicts_by_hits_freq(yuid_key, recs):
    # login with the most hits is the best
    login_hits = defaultdict(int)
    for r in recs:
        login_hits[r['login']] += 1

    max_hits = max(login_hits.iteritems(), key=lambda x: x[1])[1]
    best_logins = [login for login, c in login_hits.iteritems() if c == max_hits]

    if len(best_logins) == 1:  # only single most frequent
        yield {'login': best_logins[0], 'yuid': yuid_key['yuid'],
               'conflicts_count': len(login_hits),
               'conflicts': utils.default_to_regular(login_hits)}
    else:
        for best_login in best_logins:
            yield {'login': best_login, 'yuid': yuid_key['yuid'],
                   'max_hits': max_hits, 'conflicting_logins': best_logins,
                   '@table_index': 1}


def remove_large_rlogins(login_key, recs, max_count=100):
    l_recs = []
    count = 0
    for r in recs:
        count += 1
        if count <= max_count:
            l_recs.append(r)

    if count <= max_count:
        for r in l_recs:
            r['@table_index'] = 0
            yield r
    else:
        yield {'login': login_key['login'], 'yuids_count': count, '@table_index': 1}


def get_rlogin_and_cid_splices(rlogin_yuid_info, crypta_yuid_info,
                               record_filter=lambda record: True,
                               id_key='yuid', add_info=[]):
    # TODO: no need to return rows with all params. Only yuids are used outside
    rlogin_graph = defaultdict(list)
    for record in rlogin_yuid_info:
        if record_filter(record):
            yuid = record[id_key]
            login = record['login']
            assessor_row = {id_key: yuid,
                            'login': login}
            for key in add_info:
                assessor_row[key] = record[key]
            rlogin_graph[login].append(assessor_row)

    crypta_graph = defaultdict(list)
    for record in crypta_yuid_info:
        if record_filter(record):
            yuid = record[id_key]
            crypta_id = record['crypta_id']
            crypta_row = {id_key: yuid,
                          'crypta_id': crypta_id}
            for key in add_info:
                crypta_row[key] = record[key]
            crypta_graph[crypta_id].append(crypta_row)

    return rlogin_graph, crypta_graph


def get_device_type_crypta(ua_profile):
    if 'phone' in ua_profile:
        return 'phone'
    elif 'desk' in ua_profile:
        return 'desk'
    else:
        return 'unknown'


def distinct(key, recs, distinct_by):
    output = {}
    for distinct_key in distinct_by:
        output[distinct_key] = key[distinct_key]
    yield output


def filter_by_yuids(rec, yuids):
    out_rec = dict()

    if rec['yuid'] not in yuids:
        return

    for yuid_col in YUID_ALL_COLUMNS:
        col_value = rec.get(yuid_col)
        if col_value:
            out_rec[yuid_col] = col_value
        else:
            return

    yield out_rec


def prepare_vertices(rec, office_yuids_dict):
    id_type = rec['id_type']
    if id_type.startswith('yuid'):
        yuid = rec['key']

        if yuid in office_yuids_dict:
            out_rec = {'yuid': yuid, 'crypta_id': rec['crypta_id']}
            out_rec.update(office_yuids_dict[yuid])

            yield out_rec


def prepare_device_vertices(rec, office_devids_dict):
    id_type = rec['id_type']
    if id_type.startswith('deviceid'):
        devid = rec['key']

        if devid in office_devids_dict:
            out_rec = {'devid': devid, 'crypta_id': rec['crypta_id']}
            out_rec.update(office_devids_dict[devid])

            yield out_rec


def merge_yuid_and_devid_graphs(yuid_graph, device_graph):
    graph = defaultdict(list)
    for uid in yuid_graph:
        if uid not in device_graph:
            continue
        graph[uid] = device_graph[uid]
        device_ua_profiles = set([node['ua_profile']
                                  for node in device_graph[uid]])
        # yuid_ua_profile = set([node['ua_profile']
        #                           for node in yuid_graph[uid]])
        for node in yuid_graph[uid]:
            if node['ua_profile'] in device_ua_profiles:
                graph[uid].append(node)
    return graph


def devid_to_yuid(crypta_full_graph):
    graph = defaultdict(list)
    for key, nodes in crypta_full_graph.items():
        for node in nodes:
            if 'devid' in node:
                node['yuid'] = 'devid_' + node['devid']
            graph[key].append(node)
    return graph


def reduce_add_ua_profile(yuid_key, recs):
    lrecs = list(recs)
    yuid_recs = [r for r in lrecs if r['@table_index'] == 0]
    ua_recs = [r for r in lrecs if r['@table_index'] == 1]
    if yuid_recs and ua_recs:
        yuid_rec = yuid_recs[0]
        ua_profile = ua_recs[0]['ua_profile']
        yuid_rec['ua_profile'] = ua_profile
        yuid_rec['device_type'] = get_device_type_crypta(ua_profile)
        yield yuid_rec
