#!/usr/bin/python
# coding=utf-8
from functools import partial

import luigi
import yt.wrapper as yt

import rtcconf.config
from data_imports.import_logs import app_metrica_day
from data_imports.import_logs import graph_import_fp
from data_imports.import_logs.webvisor import graph_webvisor
from lib.luigi import base_luigi_task
from lib.luigi import yt_luigi
from matching.human_matching import graph_vertices_base
from rtcconf import config
from utils import mr_utils as mr
from v2 import ids


def add_experiment_id(rec, experiment_id):
    rec['experiment_id'] = experiment_id
    yield rec


def merge_bb_experiments(_, recs, keyword_source='390'):
    recs = list(recs)
    if len(recs) > 1:
        # Crypta ids must not intersect among experiments
        for r in recs:
            r['@table_index'] = 1
            yield r
        return

    rec = recs[0]

    crypta_id = rec['crypta_id']
    id_type = rec['id_type']
    sources = rec['sources']
    experiment_id = rec.get('experiment_id')

    if id_type.startswith('yuid'):
        yuid = rec['id']
    else:
        yuid = crypta_id  # AHAHAHAHAHAHAAAAAA

    hub_bits = convert_to_bb_hub_source_bits(id_type, sources.split(','), experiment_id)
    value = 'keyword=%s\tyuid=%s\tcid=%s\tvalue=%s' % \
            (keyword_source, yuid, crypta_id, hub_bits)

    yield {'key': yuid, 'subkey': '', 'value': value}


def merge_bb_rm_cid(_, recs, keyword_source='390'):
    recs = list(recs)
    rec = recs[0]

    id_type = rec['id_type']
    if id_type.startswith('yuid'):
        yuid = rec['id']
        crypta_id = rec['crypta_id']
        stub = '00000000000000000000000000000000'
        # sources = rec['sources']
        # value = 'keyword=%s\tyuid=%s\tcid=%s\tvalue=%s,%s' % \
        #         (keyword_source, yuid, crypta_id, sources, id_type)
        value = 'keyword=%s\tyuid=%s\tcid=%s\tvalue=%s' % \
                (keyword_source, yuid, crypta_id, stub)

        yield {'key': yuid, 'subkey': '', 'value': value}


def get_source_bits(id_type, source):
    bits_positions = set()
    indevice = False

    if id_type.startswith('yuid'):
        id_pair_type, source_type = tuple(source.split('_', 1))

        if source_type == config.ID_SOURCE_TYPE_PASSPORT or source_type == config.ID_SOURCE_TYPE_PASSPORT_SENSITIVE:
            bits_positions.add(0)
        elif source_type == config.ID_SOURCE_TYPE_WEBVISOR:
            bits_positions.add(1)
        elif id_pair_type == config.ID_TYPE_EMAIL:
            bits_positions.add(2)
        elif id_pair_type == config.ID_TYPE_FUID:
            bits_positions.add(3)
            indevice = True
        elif id_pair_type == config.ID_TYPE_BAR_UI:
            bits_positions.add(4)
            indevice = True
        # elif id_pair_type == config.ID_TYPE_EAL:
        #     bits_positions.add(5)
        #     indevice = True
        elif id_pair_type in [rtcconf.config.PAIR_SRC_FUZZY]:
            bits_positions.add(6)

        # Properly handle indevice flag for oauth_indev, yabrowser_android_indev etc.
        match_type = source.rsplit('_', 1)
        if match_type == config.INDEVICE:
            indevice = True

    else:
        source_type, match_type = tuple(source.rsplit('_', 1))

        if match_type == config.INDEVICE:
            indevice = True

        if source_type == config.ID_SOURCE_TYPE_ACCOUNT_MANAGER:
            bits_positions.add(7)
            # TODO: fetch real email/phone source from match chain
        elif source_type == config.ID_SOURCE_TYPE_REDIR:
            bits_positions.add(8)
            bits_positions.add(0)  # for login

        elif source_type in [config.ID_SOURCE_TYPE_YABROWSER_ANDROID,  # all exact in-device
                             config.ID_SOURCE_TYPE_YABROWSER_IOS,
                             config.ID_SOURCE_TYPE_WATCH_LOG,
                             config.ID_SOURCE_TYPE_ACCESS_LOG,
                             config.ID_SOURCE_TYPE_TRACK,
                             config.ID_SOURCE_TYPE_REDIR,
                             config.ID_SOURCE_TYPE_STARTUP]:
            bits_positions.add(9)
        elif source_type == config.ID_SOURCE_TYPE_FUZZY:
            bits_positions.add(10)
        elif match_type in config.NO_MATCH_TYPES:
            bits_positions.add(11)

    return bits_positions, indevice


def convert_to_bb_hub_source_bits(id_type, sources, experiment_id):
    # | 4 experiment bits | ... source bits | 1 in-device bit | 3 id type bits | = 32 bits total
    bits_total_len = 32
    experiment_bits_len = 4
    id_type_bits_len = 4  # id_type + in-device

    bits = set()

    # id types encoding
    supported_id_types = {
        'yuid-desktop': 0b001,
        'yuid-mobile': 0b010,
        'deviceid': 0b100
    }
    if id_type in supported_id_types:
        # id type bits goes with no shift
        bits.add(supported_id_types[id_type])

    # source types encoding
    sources_bits_len = bits_total_len - id_type_bits_len - experiment_bits_len  # all the rest
    # every source is represented by a single bit. It is the way to support multi-source

    for s in sources:
        bits_positions, indevice = get_source_bits(id_type, s)
        for bit_pos in bits_positions:
            if bit_pos > sources_bits_len - 1:
                raise Exception('Current bits format supports only %d bits for source encoding' % sources_bits_len)

            # start from the left after id type
            sources_shift = id_type_bits_len
            bits.add(1 << (sources_shift + bit_pos))
        if indevice:  # if at least one pair was indevice, count as indevice
            bits.add(0b1000)

    # experiment encoding
    if experiment_id:
        experiment_id_bin_str = bin(experiment_id)[2:]
        if len(experiment_id_bin_str) > experiment_bits_len:
            raise Exception(
                'Current bits format supports only %d bits for experiment encoding' % experiment_bits_len)
        # one of 4 leftmost bits
        experiment_shift = bits_total_len - experiment_bits_len
        bits.add(experiment_id << experiment_shift)

    merged_bits = 0
    for b in bits:
        merged_bits |= b

    result = format(merged_bits, '032b')
    assert (len(result) == bits_total_len)
    return result


def get_today_yuids_and_devids(graph_date_folder, vertices_folder):
    today_yuids = vertices_folder + 'yuids_ua_day_tmp'
    today_devids = vertices_folder + 'dev_info_day_tmp'
    yt.run_map(partial(mr.map_column_to_key, column='yuid'),
               graph_date_folder + 'yuids_ua_day', today_yuids)
    yt.run_map(partial(mr.map_column_to_key, column=ids.CRYPTA_DEVICE_ID),
               graph_date_folder + 'mobile/dev_info_yt', today_devids)
    mr.sort_all([today_yuids, today_devids],
                sort_by='key')
    return today_yuids, today_devids


def prepare_for_upload(vertices_config):
    graph_date_folder = config.YT_OUTPUT_FOLDER + vertices_config.date + '/'
    workdir = vertices_config.get_vertices_folder() + 'upload/'
    mr.mkdir(workdir)

    today_vertices = vertices_config.get_vertices_table()
    prev_vertices = vertices_config.get_prev_vertices_table()

    if prev_vertices:
        vertices_tables = [today_vertices, prev_vertices]
    else:
        vertices_tables = [today_vertices]

    today_yuids, today_devids = get_today_yuids_and_devids(graph_date_folder, workdir)

    out_tables = [workdir + 'vertices_add_or_update', workdir + 'vertices_remove', workdir + 'vertices_merge_overlimit']
    yt.run_reduce(graph_vertices_base.reduce_changed,
                  [today_yuids, today_devids] +
                  vertices_tables,
                  out_tables,
                  reduce_by='key')

    mr.merge_chunks_all(out_tables)

    yt.remove(today_yuids)
    yt.remove(today_devids)


class PrepareVerticesForUpload(base_luigi_task.BaseTask):
    vertices_config = luigi.Parameter()

    def requires(self):
        return [
            self.vertices_config.producing_task,
            app_metrica_day.ImportAppMetrikaDayTask(date=self.vertices_config.date, run_date=self.vertices_config.date),
            graph_import_fp.ImportFPDayTask(date=self.vertices_config.date, run_date=self.vertices_config.date),
            graph_webvisor.ImportWebvisorTask(date=self.vertices_config.date, run_date=self.vertices_config.date),
        ]

    def run(self):
        prepare_for_upload(self.vertices_config)
        mr.set_generate_date(self.vertices_config.get_upload_table(), self.vertices_config.date)
        mr.set_generate_date(self.vertices_config.get_upload_rm_table(), self.vertices_config.date)

    def output(self):
        # YtDateTarget is used because new fuzzy vertices are in folder with no date
        return [yt_luigi.YtDateTarget(self.vertices_config.get_upload_table(), self.vertices_config.date),
                yt_luigi.YtDateTarget(self.vertices_config.get_upload_rm_table(), self.vertices_config.date)]


if __name__ == '__main__':
    pass
