from collections import Counter

import yt.wrapper as yt

import utils
from rtcconf import config

OBL_LEVEL = 5
CITY_LEVEL = 6


class LookupWithStats(object):
    """
        Counts number of times geobase4.Lookup is accessed
    """

    def __init__(self, lookup):
        self.lookup = lookup
        self.stats = dict(is_in=0, region=0)

    def is_in(self, ip, pid):
        self.stats['is_in'] += 1
        return self.lookup.is_in(ip, pid)

    def region(self, ip):
        self.stats['region'] += 1
        return self.lookup.region(ip)

    def reset_stats(self):
        for k in self.stats:
            self.stats[k] = 0

    def find_country_id(self, reg_id):
        return self.lookup.find_country_id(reg_id)

    def region_by_id(self, reg_id):
        return self.lookup.region_by_id(reg_id)


def mk_mapper_with_geobase_stats(real_mapper):
    @yt.aggregator
    def fake_mapper(recs):
        for rec in recs:
            for res in real_mapper(rec):
                yield res
        write_geobase_stats()

    return fake_mapper


def geodata_region(ip):
    l = get_geo_lookup(use_local_bin_file=False)
    try:
        return l.region(ip)
    except ValueError:
        return None


def write_geobase_stats():
    stats = dict()
    geofuncs = [geodata_all_ip_rus, geodata_region]
    for f in geofuncs:
        if hasattr(f, 'l') and hasattr(f.l, 'stats'):
            stats.update(f.l.stats)
            f.l.reset_stats()

    if bool(stats):
        yt.write_statistics(dict(geobase=stats))


def mk_reducer_with_geobase_stats(real_reducer):
    @yt.reduce_aggregator
    def fake_reducer(pairs):
        for pair in pairs:
            for rec in real_reducer(pair[0], pair[1]):
                yield rec
        write_geobase_stats()

    return fake_reducer


def get_region_info(reg_id, l):
    country_id = None
    oblast_id = None
    city_id = None

    try:
        country_id = l.find_country_id(reg_id) or None  # country has it's own api level

        # trying to find city and oblast region up in region tree
        current_level_reg_id = reg_id
        for step in range(10):  # max depth
            current_level_reg = l.region_by_id(current_level_reg_id)
            level_type = current_level_reg.type
            if level_type < OBL_LEVEL:
                break

            if level_type == OBL_LEVEL:
                oblast_id = current_level_reg_id
            elif level_type == CITY_LEVEL:
                city_id = current_level_reg_id

            current_level_reg_id = current_level_reg.parent
    except RuntimeError:
        pass

    main_region_id = oblast_id or city_id

    return country_id, oblast_id, city_id, main_region_id


def get_geo_lookup(use_local_bin_file=True):
    if not hasattr(get_geo_lookup, 'l'):
        from geobase5 import Lookup
        if use_local_bin_file:
            get_geo_lookup.l = LookupWithStats(Lookup('/var/cache/geobase/geodata4.bin'))
        else:
            # should be mapped using yt_files operation param
            get_geo_lookup.l = LookupWithStats(Lookup('geodata4.bin'))

    return get_geo_lookup.l


def geodata_all_ip_rus(ips):
    l = get_geo_lookup(use_local_bin_file=False)
    try:
        return all([l.is_in(ip, 225) for ip in ips])
    except ValueError:
        return False


def get_top_by_region_level(reg_activity):
    country_counter = Counter()
    obl_counter = Counter()
    city_counter = Counter()
    main_reg_counter = Counter()

    for reg_id, hits in reg_activity.iteritems():
        try:
            reg_id = int(reg_id)
        except ValueError:
            pass

        country_id, oblast_id, city_id, main_region = get_region_info(
            reg_id, get_geo_lookup(use_local_bin_file=False)
        )
        country_counter[country_id] += hits
        obl_counter[oblast_id] += hits
        city_counter[city_id] += hits
        main_reg_counter[main_region] += hits

    return (
        utils.top(country_counter),
        utils.top(obl_counter),
        utils.top(city_counter),
        utils.top(main_reg_counter),
    )


def geodata_yt_table(dt):
    geodata = '//statbox/statbox-dict-by-name/geodata4.bin/%s' % dt
    if not yt.exists(geodata):
        geodata = '//statbox/statbox-dict-last/geodata4.bin'
    return yt.TablePath(geodata, attributes={'file_name': 'geodata4.bin'})


def get_local_geodata4_bin_path():
    import os
    if hasattr(config, 'LOCAL_GEODATA4_BIN'):
        return config.LOCAL_GEODATA4_BIN
    else:
        return os.getenv('LOCAL_GEODATA4_BIN', '/var/cache/geobase/geodata4.bin')


def get_europe_and_america_regs():
    from geobase4 import Lookup
    l = Lookup(get_local_geodata4_bin_path())
    europe = set([r.id for r in l.tree(111)])
    europe.add(111)
    america = set([r.id for r in l.tree(10002)])
    america.add(10002)
    return europe | america
