import uatraits
import re


def singleton(class_):
    instances = {}

    def getinstance(*args, **kwargs):
        if class_ not in instances:
            instances[class_] = class_(*args, **kwargs)
        return instances[class_]

    return getinstance


@singleton
class UaDetect(object):
    detector = uatraits.detector('/usr/share/uatraits/browser.xml', '/usr/share/uatraits/profiles.xml')


def _get_platform(profile_info):
    vendor = profile_info.get('DeviceVendor', '').lower()
    os = profile_info.get('OSFamily', '').lower()
    osversion = profile_info.get('OSVersion', '').lower()

    platform_type = 'd|desk'
    if profile_info.get('isMobile', ''):
        platform_type = 'm|phone'
    if profile_info.get('isTablet', ''):
        platform_type = 'm|tablet'
    if profile_info.get('isTV', ''):
        platform_type = 'd|tv'

    if platform_type.startswith('d'):
        platform, version = os, osversion
    else:
        platform, version = _get_mobile_platform(os, osversion, vendor)

    return platform_type, platform, version


def _get_mobile_platform(app_platform, os_version, vendor):
    if not app_platform or not os_version:
        return '', ''

    elif "iphone" in app_platform:
        app_platform = "ios"
    elif "ipad" in app_platform:
        app_platform = "ios"
    elif "ios" in app_platform:
        app_platform = "ios"
    elif "android" in app_platform:
        app_platform = "android"
    elif "apad" in app_platform:
        app_platform = "android"
    elif "windowsmobile" == app_platform:
        app_platform = "windows"
    elif "windowsphone" == app_platform:
        app_platform = "windows"
    elif "windows phone" == app_platform:
        app_platform = "windows"
    elif "wp" == app_platform:
        app_platform = "windows"
    elif "winrt" == app_platform:
        app_platform = "windows"
    elif "bada" == app_platform:
        app_platform = "bada"
    elif "tizen" == app_platform:
        app_platform = "tizen"
    else:
        return '', ''

    if app_platform == 'windows':
        os_version = _get_first_digit_version(os_version)
    else:
        os_version = _get_dot_version(os_version)

    return _get_vendor(vendor) + '|' + app_platform, os_version


def _get_dot_version(os_version):
    m = re.search('.*?(\d+\.\d+\.\d+).*', os_version)
    if m:
        return m.group(1)
    m = re.search('.*?(\d+\.\d+).*', os_version)
    if m:
        return m.group(1)
    m = re.search('.*?(\d+).*', os_version)
    if m:
        return m.group(1)
    return ''


def _get_first_digit_version(os_version):
    return os_version[0]


def _get_vendor(vendor):
    if 'apple' in vendor:
        return 'apple'
    if 'samsung' in vendor:
        return 'samsung'
    if vendor.startswith('htc'):
        return 'htc'
    if 'huawei' in vendor:
        return 'huawei'
    if vendor.startswith('acer'):
        return 'acer'
    if 'nokia' in vendor:
        return 'nokia'
    if 'lenovo' in vendor:
        return 'lenovo'
    if vendor.startswith('lg'):
        return 'lg'
    if 'megafon' in vendor:
        return 'megafon'
    if vendor.startswith('dell'):
        return 'dell'
    if vendor.startswith('fly'):
        return 'fly'
    if vendor.startswith('asus'):
        return 'asus'
    if vendor.startswith('hp'):
        return 'hp'
    if 'alcatel' in vendor:
        return 'alcatel'
    if 'archos' in vendor:
        return 'archos'
    if 'garmin' in vendor:
        return 'garmin'
    if 'gm' == vendor:
        return 'gm'
    if vendor.startswith('haier'):
        return 'haier'
    if vendor.startswith('hw'):
        return 'hw'
    if 'hyundai' in vendor:
        return 'hyundai'
    if 'mediatek' in vendor:
        return 'mediatek'
    if 'micromax' in vendor:
        return vendor
    if 'motorola' in vendor:
        return 'motorola'
    if vendor.startswith('onda'):
        return 'onda'
    if vendor.startswith('orange'):
        return 'orange'
    if 'panasonic' in vendor:
        return 'panasonic'
    if vendor.startswith('pantech'):
        return 'pantech'
    if vendor.startswith('senseit'):
        return 'senseit'
    if vendor.startswith('sharp'):
        return 'sharp'
    if 'sony' in vendor:
        return 'sony'
    if vendor.startswith('spreadtrum'):
        return 'spreadtrum'
    if vendor.startswith('tcl'):
        return 'tcl'
    if vendor.startswith('unimax'):
        return 'unimax'
    if vendor.startswith('vertu'):
        return 'vertu'
    if 'vodafone' in vendor:
        return 'vodafone'
    if vendor.startswith('wexler'):
        return 'wexler'
    if vendor.startswith('zte') or vendor == 'uzte':
        return 'zte'
    if 'google' in vendor:
        return 'google'

    if vendor.startswith('xiaomi'):
        return 'xiaomi'
    if vendor.startswith('discovery'):
        return 'discovery'
    if 'prestigio' in vendor:
        return 'prestigio'
    if vendor.startswith('explay'):
        return 'explay'
    if vendor.startswith('dexp'):
        return 'dexp'
    if vendor == 'general mobile' or vendor == 'general_mobile' or vendor == 'general+mobile' or \
                    vendor == 'generalmobile':
        return 'general_mobile'
    if 'meizu' in vendor:
        return 'meizu'
    if 'philips' in vendor:
        return 'philips'
    if vendor.startswith('qmobile'):
        return 'qmobile'
    if vendor.startswith('vivo'):
        return 'vivo'
    if vendor.startswith('highscreen'):
        return 'highscreen'
    if 'turkcell' in vendor:
        return 'turkcell'
    if vendor.startswith('mtc'):
        return 'mtc'
    if 'doogee' in vendor:
        return 'doogee'
    if vendor == 'lava':
        return 'lava'
    if vendor.startswith('tecno'):
        return 'tecno'
    if 'beeline' in vendor:
        return 'beeline'
    if 'oysters' in vendor:
        return 'oysters'
    if vendor.startswith('casper'):
        return 'casper'
    if vendor.startswith('oneplus'):
        return 'oneplus'
    if vendor.startswith('texet'):
        return 'texet'
    if vendor.startswith('irbis'):
        return 'irbis'
    if vendor == 'bq' or vendor.startswith('bq ') or vendor.startswith('bqs-') or vendor.startswith('bq-'):
        return 'bq'
    if vendor.startswith('arkelectronic') or vendor.startswith('ark-electronic') or \
            vendor.startswith('ark+electronic') or vendor.startswith('ark benefit'):
        return 'ark'
    if vendor.startswith('infinix'):
        return 'infinix'
    if vendor == 'digma':
        return 'digma'
    if vendor == 'dns' or vendor.startswith('dns ') or vendor.startswith('dns_'):
        return 'dns'
    if vendor.startswith('oppo'):
        return 'oppo'
    if 'andromax' in vendor:
        return 'andromax'
    if vendor == 'lanix':
        return 'lanix'
    if 'coolpad' in vendor:
        return 'coolpad'
    if 'microsoft' in vendor:
        return 'microsoft'
    if vendor.startswith('myphone'):
        return 'myphone'
    if vendor.startswith('supra ') or vendor == 'supra':
        return 'supra'
    if vendor.startswith('symphony'):
        return 'symphony'
    if vendor.startswith('4good'):
        return '4good'
    if vendor.startswith('tele2'):
        return 'tele2'
    if 'jiayu' in vendor:
        return 'jiayu'
    if vendor.startswith('azumi'):
        return 'azumi'
    if vendor == 'nomi' or vendor.startswith('nomi ') or vendor.startswith('nomi_'):
        return 'nomi'
    if vendor == 'amazon':
        return 'amazon'
    if 'keneksi' in vendor:
        return 'keneksi'
    if vendor == 'hisense':
        return 'hisense'

    return 'un'


def _get_ua_profile(profile_info):
    return '|'.join(_get_platform(profile_info))


def encode_startup(app_platform, os_version, vendor, device_type, model):
    os = (app_platform or '').lower()
    osversion = (os_version or '').lower()
    vendor = (vendor or '').lower()
    model = (model or '').lower()

    # For some reason lg nexus in uatraites has google as vendor
    if 'nexus' in model and vendor == 'lg':
        vendor = 'google'
    platform, version = _get_mobile_platform(os, osversion, vendor)
    return 'm|' + (device_type or '') + '|' + platform + '|' + version


class Ua(object):
    def __init__(self, user_agent, wapprofile='', xoperaminiphoneua='', xwapprofile=''):
        self.user_agent = user_agent
        detector = UaDetect().detector
        if xoperaminiphoneua:
            user_agent = xoperaminiphoneua
        if xwapprofile:
            wapprofile = xwapprofile
        if wapprofile and user_agent:
            headers = {'X-Wap-Profile': wapprofile, 'User-Agent': user_agent}
            profile_info = detector.detect_by_headers(headers)
        else:
            profile_info = detector.detect(user_agent)
        self.profile_info = profile_info

    def is_bad(self):
        appmetrica = self.user_agent.startswith('com.yandex.mobile.metrica.ads.sdk')
        return appmetrica or (not self.is_browser()) or self.is_emulator() or self.is_robot()

    def is_browser(self):
        return self.profile_info.get('isBrowser', True)

    def is_webview(self):
        return self.profile_info.get('inAppBrowser', False)

    def is_emulator(self):
        return self.profile_info.get('isEmulator', False)

    def is_robot(self):
        return self.profile_info.get('isRobot', False)

    def is_mobile(self):
        return self.profile_info.get('isMobile', False)

    def get_browser(self):
        browser_name = self.profile_info.get('BrowserName', '').lower()
        browser_version = self.profile_info.get('BrowserVersion', '').lower()
        return browser_name, browser_version

    def to_ua_profile(self):
        return _get_ua_profile(self.profile_info)
