import logging

import luigi
import yt.wrapper as yt

from lib.luigi import yt_luigi
from matching.device_matching.app_metrica import app_metrica_month
from rtcconf import config
from utils import mr_utils as mr
from v2 import ids


def map_yuid_with_all(rec):
    # map to java format
    out_rec = {'id': rec['yuid'],
               'id_type': ids.YANDEXUID,
               'source': "yuid_with_all",
               'dates': rec['all_ip_dates']}

    for col in [
        'yuid_creation_date',
        'ip_activity_type',

        'browser',
        'browser_version',
        'ua_profile',
        'ua',
        'webview',

        'gender',
        'age_segments',
        'income_segments',

        'main_region',
        'main_region_country',
        'main_region_obl',
        'main_region_city'
    ]:
        out_rec[col] = rec[col]

    yield out_rec

yuid_info_schema = {
    'id': 'string',
    'id_type': 'string',
    'source': 'string',
    'yuid_creation_date': 'string',
    'ip_activity_type': 'string',
    'dates': 'any',

    'browser': 'string',
    'browser_version': 'string',
    'ua_profile': 'string',
    'ua': 'string',
    'webview': 'boolean',

    'gender': 'any',
    'age_segments': 'any',
    'income_segments': 'any',

    'main_region': 'uint64',
    'main_region_country': 'uint64',
    'main_region_obl': 'uint64',
    'main_region_city': 'uint64'
}

class YuidWithAllToIdsStorage(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {
            'dict': config.GRAPH_YT_DICTS_FOLDER
        }

    def output_folders(self):
        return {
            'ids_storage': config.CRYPTA_IDS_STORAGE,
        }

    def requires(self):
        from matching.yuid_matching import graph_dict
        return [graph_dict.YuidAllIdDictsTask(self.date)]

    def run(self):
        dict_f = self.in_f('dict')
        ids_storage_f = self.out_f('ids_storage')
        mr.mkdir(ids_storage_f)
        mr.mkdir(ids_storage_f + ids.YANDEXUID)

        out_table = ids_storage_f + ids.YANDEXUID + '/yuid_with_all_info'

        with yt.Transaction() as tr:
            mr.create_table_with_schema(
                out_table,
                yuid_info_schema,
                tr,
                strict=True,
                recreate_if_exists=True
            )

            yt.run_map(map_yuid_with_all,
                       dict_f + 'yuid_with_all',
                       out_table)

            mr.sort_all([
                out_table,
            ], sort_by=['id', 'id_type'])

        mr.set_generate_date(out_table, self.date)


    def output(self):
        ids_storage_f = self.out_f('ids_storage')
        return [
            yt_luigi.YtDateTarget(ids_storage_f + ids.YANDEXUID + '/yuid_with_all_info', self.date),
        ]


class IdsStorageIsReady(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        return [YuidWithAllToIdsStorage(self.date),
                app_metrica_month.AppMetricaDictMergeMonthTask(self.date)]


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)
    yt.config["tabular_data_format"] = yt.YsonFormat(process_table_index=True)
    logging.basicConfig(level='INFO')

    task = YuidWithAllToIdsStorage('2017-04-12')

    # import graph_stat_vertices
    #
    # r = graph_stat_vertices.VerticesAllStatsTask([task.vertices_config])
    #
    luigi.build([task], workers=5, scheduler_port=8083)
