import luigi

from lib.luigi import base_luigi_task
from lib.luigi import yt_luigi
from lib.sandbox.api import SandboxApi
from lib.sandbox.env import get_appropriate_sandbox_env
from rtcconf import config
from v2.ids_storage import dicts_to_storage
from v2.soup import graph_soup


class BaseSandboxTask(base_luigi_task.BaseTask):
    date = luigi.Parameter()

    def __init__(self, date, sandbox_task_type, sandbox_task_params):
        self.sandbox_task_type = sandbox_task_type
        self.sandbox_task_params = sandbox_task_params

        super(BaseSandboxTask, self).__init__(date)

    def run(self):
        sandbox_api = SandboxApi(config.SANDBOX_OAUTH)

        timeout_hours = 20

        task_id = sandbox_api.start_task_async(
            self.sandbox_task_type,
            params={
                'owner': 'CRYPTA',
                'priority': {
                    'class': 'SERVICE',
                    'subclass': 'NORMAL'
                },
                'kill_timeout': timeout_hours * 60 * 60
            },
            custom_params=self.sandbox_task_params
        )

        sandbox_api.wait_for_task(task_id, timeout_in_hours=timeout_hours)

        yt_luigi.TodayFileTarget.done(config.LOCAL_OUTPUT_FOLDER + self.sandbox_task_type, self.date)

    def output(self):
        return yt_luigi.TodayFileTarget(config.LOCAL_OUTPUT_FOLDER + self.sandbox_task_type, self.date)


class PrepareSoupForMatching(BaseSandboxTask):
    date = luigi.Parameter()

    def requires(self):
        return [
            dicts_to_storage.IdsStorageIsReady(self.date),
            graph_soup.SoupIsReady(self.date)
        ]

    def __init__(self, date):
        sandbox_task_type = 'CRYPTA_GRAPH_PREPARE_SOUP_YAML'
        sandbox_task_params = {
            'environment': get_appropriate_sandbox_env(),
            'soupGenerationDate': date
        }
        super(PrepareSoupForMatching, self).__init__(
            date=date,
            sandbox_task_type=sandbox_task_type,
            sandbox_task_params=sandbox_task_params
        )


class GraphV2HumanMatching(BaseSandboxTask):
    date = luigi.Parameter()

    def requires(self):
        return [
            PrepareSoupForMatching(self.date)
        ]

    def __init__(self, date):
        sandbox_task_type = 'CRYPTA_GRAPH_HUMAN_MATCHING_YAML'
        sandbox_task_params = {
            'environment': get_appropriate_sandbox_env()
        }
        super(GraphV2HumanMatching, self).__init__(
            date=date,
            sandbox_task_type=sandbox_task_type,
            sandbox_task_params=sandbox_task_params
        )


class GraphV2DirectMatching(BaseSandboxTask):
    date = luigi.Parameter()

    def requires(self):
        return [
            PrepareSoupForMatching(self.date)
        ]

    def __init__(self, date):
        sandbox_task_type = 'CRYPTA_GRAPH_DIRECT_MATCHING_YAML'
        sandbox_task_params = {
            'environment': get_appropriate_sandbox_env()
        }
        super(GraphV2DirectMatching, self).__init__(
            date=date,
            sandbox_task_type=sandbox_task_type,
            sandbox_task_params=sandbox_task_params
        )


if __name__ == '__main__':
    # import yt.wrapper as yt
    import logging
    logging.basicConfig(level="INFO")

    sandbox_api = SandboxApi()

    timeout_hours = 20
    sandbox_task_params = {
        'environment': 'testing'
    }

    task_id = sandbox_api.start_task_async(
        'CRYPTA_GRAPH_PREPARE_SOUP_YAML',
        params={
            'owner': 'CRYPTA',
            'priority': {
                'class': 'SERVICE',
                'subclass': 'NORMAL'
            },
            'kill_timeout': timeout_hours * 60 * 60
        },
        custom_params=sandbox_task_params
    )

    sandbox_api.wait_for_task(task_id, timeout_in_hours=timeout_hours)
