from functools import partial

import yt.wrapper as yt

from utils import mr_utils as mr
from lib.luigi import yt_luigi
from rtcconf import config
from v2.soup import soup_config, soup_edge_type

edge_key_columns = ['id1Type', 'id2Type', 'sourceType', 'logSource', 'id1', 'id2']
edge_type_key_columns = ['id1Type', 'id2Type', 'sourceType', 'logSource']


class SoupDailyLogTable(object):
    """
    Intermediate tmp table between transactional log parsing and daily storage soup table
    """

    def __init__(self, log_source, date):
        self.log_source = log_source
        self.date = date
        self.soup_daily_tables = [
            SoupDailyTable(et, self.date)
            for et in soup_config.ALL_EDGES_BY_LOG_DICT[log_source]
            if et.supply_type == soup_edge_type.SupplyType.DAILY
        ]

    @property
    def schema(self):
        return {'id1Type': 'string', 'id2Type': 'string',
                'sourceType': 'string', 'logSource': 'string',
                'id1': 'string', 'id2': 'string',
                'hits': 'uint64', 'ts': 'int64'}

    def dir_path(self):
        return soup_config.SOUP_DAY_LOGS_DIR + self.date + '/'

    def ensure_dir(self):
        mr.mkdir(self.dir_path())

    def table_path(self):
        return self.dir_path() + self.log_source

    def create(self):
        path = self.table_path()
        if self.log_source == soup_config.LOG_SOURCE_WATCH_LOG:
            # same crutch as in finalize fun vvv
            path += '_py'
        with yt.Transaction() as tr:
            mr.create_table_with_schema(
                path,
                self.schema,
                tr,
                strict=True,
                recreate_if_exists=True
            )
        return path

    def finalize(self):
        if self.log_source == soup_config.LOG_SOURCE_WATCH_LOG:
            # map will overwrite table data if exists, so fill
            # ...soup/wl in yql and ...soup/wl_py in python
            # and map from 2 tables, and remove 2 tables too
            # TODO: delete this crutch when metrica socke come to yql
            yt.remove(self.table_path() + '_py')
        yt.remove(self.table_path())

    def prepare_daily_tables_from_log(self):
        mr.mkdir(soup_config.SOUP_DAY_DIR + self.date)

        with yt.Transaction() as tr:
            log_table = self.table_path()

            if self.log_source == soup_config.LOG_SOURCE_WATCH_LOG:
                # map from two tables /wl and /wl_py
                # TODO: delete this crutch (see comments finalize fun ^^^)
                log_table = [log_table, log_table + '_py']

            out_table_paths = [t.create(tr) for t in self.soup_daily_tables]
            out_table_indexes = {t.edge_type: idx for idx, t in enumerate(self.soup_daily_tables)}

            errors_table = soup_config.SOUP_DAY_DIR + self.date + '/' + self.log_source + '_errors'
            errors_idx = len(out_table_indexes)
            out_table_indexes['errors'] = errors_idx

            if isinstance(log_table, list):
                partition_count = (
                    mr.calculate_optimized_mr_partition_count(log_table[0])
                    + mr.calculate_optimized_mr_partition_count(log_table[1]))
            else:
                partition_count = mr.calculate_optimized_mr_partition_count(log_table)

            from v2.soup import soup_utils

            yt.run_map_reduce(
                None,
                partial(
                    soup_utils.reduce_day_activity,
                    date=self.date,
                    out_table_indexes=out_table_indexes,
                    key_weight_limit=config.YT_KEY_SIZE_LIMIT
                ),
                log_table,
                out_table_paths + [errors_table],
                reduce_by=edge_key_columns,
                spec={'partition_count': partition_count}
            )
            self.finalize()
            SoupDailyTable.finalize_all(self.soup_daily_tables, tr)

    @staticmethod
    def make_rec(id1, id2, edge_type, ts=None, hits=None, table_index=0):
        out_rec = {'id1': id1, 'id1Type': edge_type.id1_type,
                   'id2': id2, 'id2Type': edge_type.id2_type,
                   'sourceType': edge_type.source, 'logSource': edge_type.log_source,
                   'ts': ts, 'hits': hits, '@table_index': table_index}
        return out_rec

    def daily_tables_targets(self):
        return [t.as_target() for t in self.soup_daily_tables]

    def _get_processing_status_table(self):
        return soup_config.SOUP_DAY_LOGS_DIR + 'processing_status/' + self.log_source

    def set_last_processed_log_table(self, t):
        status_table = self._get_processing_status_table()
        yt.create_table(status_table, recursive=True, ignore_existing=True)
        yt.set_attribute(status_table, "last_processed_log_table", t)

    def get_last_processed_log_table(self):
        status_table = self._get_processing_status_table()
        if yt.exists(status_table):
            return yt.get_attribute(status_table, 'last_processed_log_table')
        else:
            return None


class SoupTable(object):
    """
    Base class for all soup storage tables
    """

    def __init__(self, edge_type, date):
        self.date = date
        self.edge_type = edge_type

    @property
    def schema(self):
        raise NotImplementedError

    def dir_path(self):
        raise NotImplementedError

    def ensure_dir(self):
        mr.mkdir(self.dir_path())

    def table_path(self):
        return self.dir_path() + self.edge_type.name()

    def create(self, transaction, recreate_if_exists=True):
        path = self.table_path()
        mr.create_table_with_schema(
            path,
            self.schema,
            transaction,
            strict=True,
            recreate_if_exists=recreate_if_exists
        )
        return path

    def as_target(self):
        return yt_luigi.YtDateTarget(self.table_path(), self.date)

    def finalize(self, transaction):
        SoupDumpTable.finalize_all([self], transaction)

    @staticmethod
    def finalize_all(tables, transaction):
        mr.merge_chunks_all([t.table_path() for t in tables])
        for t in tables:
            mr.set_generate_date(t.table_path(), t.date)


class SoupStorageTable(SoupTable):
    """
    Table that stores final soup aggregate
    """

    @property
    def schema(self):
        return {'id1Type': 'string', 'id2Type': 'string',
                'sourceType': 'string', 'logSource': 'string',
                'id1': 'string', 'id2': 'string', 'dates': 'any'}

    def dir_path(self):
        return soup_config.SOUP_DIR

    @staticmethod
    def make_rec(id1, id2, edge_type, dates, table_index=0):
        return {'id1Type': edge_type.id1_type, 'id2Type': edge_type.id2_type,
                'sourceType': edge_type.source, 'logSource': edge_type.log_source,
                'id1': id1, 'id2': id2,
                'dates': dates, '@table_index': table_index}


class SoupDailyTable(SoupTable):
    """
    Table stores transactional data per day
    """

    @property
    def schema(self):
        return {'id1Type': 'string', 'id2Type': 'string',
                'sourceType': 'string', 'logSource': 'string',
                'id1': 'string', 'id2': 'string',
                'date': 'string', 'dayActivity': 'any', 'dayHits': 'uint64'}

    def dir_path(self):
        return soup_config.SOUP_DAY_DIR + self.date + '/'

    def corresponding_storage_table(self):
        return SoupStorageTable(self.edge_type, self.date)

    @staticmethod
    def make_rec(id1, id2, edge_type, date, day_activity, day_hits, table_index=0):
        return {'id1Type': edge_type.id1_type, 'id2Type': edge_type.id2_type,
                'sourceType': edge_type.source, 'logSource': edge_type.log_source,
                'id1': id1, 'id2': id2,
                'date': date, 'dayActivity': day_activity, 'dayHits': day_hits, '@table_index': table_index}


class SoupDumpTable(SoupStorageTable):
    """
    Table stores intermediate data imported as dump in the same format as in storage
    """

    def dir_path(self):
        return soup_config.SOUP_DUMPS_DIR

    def corresponding_storage_table(self):
        return SoupStorageTable(self.edge_type, self.date)
