import datetime
from collections import defaultdict

from utils import utils
from v2.soup import soup_edge_type, soup_validation
from v2.soup.soup_tables import SoupDumpTable, SoupDailyTable


def get_rec_edge_type(rec):
    """
    :rtype soup.soup_edge_type.EdgeType
    """
    return soup_edge_type.get_edge_type(rec['id1Type'], rec['id2Type'], rec['sourceType'], rec['logSource'])


def _is_key_weight_ok(edge_type, id1, id2, key_weight_limit=16384):
    weight = 0
    weight += len(edge_type.id1_type)
    weight += len(edge_type.id2_type)
    weight += len(edge_type.source)
    weight += len(edge_type.log_source)
    weight += (len(id1) + len(id2))
    return weight <= key_weight_limit


def validate_dump_rec(rec, error_table_index):
    normalized_rec = soup_validation.normalize_rec(rec)
    if normalized_rec:
        yield normalized_rec
    else:
        rec['@table_index'] = error_table_index
        yield rec


# ===reduce ops===
def reduce_day_activity(edge_key, recs, date, out_table_indexes, key_weight_limit=16384):
    edge_type = get_rec_edge_type(edge_key)
    day_hits = 0
    day_activity = defaultdict(int)
    for r in recs:
        hits_count = r.get('hits') or 1
        day_hits += hits_count

        rec_ts = r.get('ts')
        if rec_ts:  # source may contain or not contain hits info
            rec_time = datetime.datetime.fromtimestamp(rec_ts)
            rec_date = utils.ts_to_date_str(rec_ts)

            if rec_date != date:
                # track when ts doesn't correspond to date (only if date is know and specified outside)
                day_activity[None] += hits_count
            else:
                day_activity[rec_time.hour] += hits_count

    day_activity = utils.default_to_regular(day_activity) if day_activity else None

    table_index = 0
    if out_table_indexes:
        table_index = out_table_indexes[edge_type]

    if not _is_key_weight_ok(edge_type, edge_key['id1'], edge_key['id2'], key_weight_limit):
        table_index = out_table_indexes['errors']

    yield SoupDailyTable.make_rec(edge_key['id1'], edge_key['id2'], edge_type,
                                  date, day_activity, day_hits, table_index)


def increment_day_activity(edge_key, recs, out_table_indexes, throw_before_date):
    all_dates = set()
    for r in recs:
        prev_all_dates = r.get('dates')
        if prev_all_dates:  # all table
            all_dates.update(prev_all_dates)

        today_date = r.get('date')
        if today_date:  # day table
            all_dates.add(today_date)

    edge_type = get_rec_edge_type(edge_key)
    table_index = 0
    if out_table_indexes:
        table_index = out_table_indexes[edge_type]

    keep_rec = True
    if throw_before_date:
        # only for daily sources
        if all_dates and all(dt < throw_before_date for dt in all_dates):
            keep_rec = False

    if keep_rec:
        out_rec = SoupDumpTable.make_rec(edge_key['id1'], edge_key['id2'], edge_type, sorted(all_dates), table_index)

        normalized_out_rec = soup_validation.normalize_rec(out_rec)
        if normalized_out_rec:
            yield normalized_out_rec

        else:
            out_rec['@table_index'] = len(out_table_indexes)  # error
            yield out_rec
