from binascii import crc32
import luigi

from ids_storage.dicts_to_storage import IdsStorageIsReady
from soup.graph_soup import SoupIsReady
from soup import soup_config

from lib.luigi.yt_luigi import BaseYtTask, YtTarget, YtAttrTarget
from lib.luigi.base_luigi_task import BaseTask
from rtcconf import config

import yt.wrapper as yt

from utils.yql_utils import run_yql


APPS_KEYWORD_ID = '723'


def appid_crc32(appid, platform):
    platform = platform.lower()
    if platform == 'ios':
        appstore = 'App Store'
    elif platform == 'android':
        appstore = 'Google Play'
    else:
        appstore = 'Bells&Whistles'

    crcstr = appid + appstore
    return crc32(crcstr) & 0xFFFFFFFF


def map_prepare_upload(rec):
    yuid = rec['yuid']
    apps = rec['apps']
    platform = rec['platform']

    apps_uints = ':'.join([str(appid_crc32(x, platform)) for x in apps])

    yield dict(
        key=yuid,
        subkey='',
        value='keyword=%s\tyuid=%s\tvalue=%s' % (APPS_KEYWORD_ID, yuid, apps_uints)
    )


class YuidAppsTask(BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {
            'storage': config.CRYPTA_IDS_STORAGE,
            'soup': soup_config.SOUP_DIR
        }

    def output_folders(self):
        return {
            'graph': config.YT_OUTPUT_FOLDER + self.date + '/'
        }

    def requires(self):
        return [IdsStorageIsReady(self.date),
                SoupIsReady(self.date)]

    def output(self):
        return YtTarget(self.out_f('graph') + 'yuid_apps')

    def run(self):
        run_yql('YuidApps', dict(date=self.date), {
            'GRAPH_YT_OUTPUT_FOLDER': config.YT_OUTPUT_FOLDER,
            'CRYPTA_IDS_STORAGE': config.CRYPTA_IDS_STORAGE,
        })


class PrepareUploadYuidAppsTask(BaseTask):
    date = luigi.Parameter()

    def _get_input_table(self):
        return YuidAppsTask(self.date).output().table

    def _get_output_table(self):
        return self._get_input_table() + '_upload'

    def requires(self):
        return YuidAppsTask(self.date)

    def output(self):
        return YtTarget(self._get_output_table())

    def run(self):
        yt.run_map(map_prepare_upload, self._get_input_table(), self._get_output_table())


class UploadYuidAppsTask(BaseTask):
    date = luigi.Parameter()

    def _upload_table(self):
        return PrepareUploadYuidAppsTask(self.date).output().table

    def requires(self):
        return PrepareUploadYuidAppsTask(self.date)

    def output(self):
        return YtAttrTarget(self._upload_table(), 'uploaded')

    def run(self):
        from matching.export.graph_upload_bb import upload_bb
        upload_bb(self._upload_table(), 'additional-apps')
        self.output().complete()