#include "parse_raw_bindings_job.h"
#include "parse_raw_bindings_mapper.h"
#include "raw_bindings_errors_schema.h"
#include "raw_bindings_parse_utils.h"

#include <crypta/dmp/adobe/bin/common/list_utils.h>
#include "crypta/dmp/adobe/bin/common/parsed_iter_bindings_schema.h"
#include <crypta/dmp/adobe/bin/common/sync_mode.h>
#include <crypta/dmp/adobe/bin/common/upload_attrs_fields.h>
#include <crypta/dmp/common/data/bindings_schema.h>

#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_suffix_generator.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>
#include <util/string/split.h>


using namespace NCrypta::NDmp::NAdobe;


namespace {
    using TParseRawIterBindingsMapper = TParseRawBindingsMapper<ConvertRawBindingsDiffToNode>;
    using TParseFullIterBindingsMapper = TParseRawBindingsMapper<ConvertRawBindingsToNode>;

    template<typename TMapper>
    void RunMap(NYT::IClientBasePtr tx, const NYT::TYPath& srcTablePath, const NYT::TRichYPath& outputTable, const NYT::TRichYPath& errorsTable, ui64 uploadTimestamp) {
        NParseRawBindingsMapper::TOutputIndexes::TBuilder outputBuilder;
        outputBuilder.Add(outputTable, NParseRawBindingsMapper::EOutputTables::Bindings);
        outputBuilder.Add(errorsTable, NParseRawBindingsMapper::EOutputTables::Errors);

        NYT::TMapOperationSpec spec;
        spec.AddInput<NYT::TNode>(srcTablePath);
        NCrypta::AddOutputs<NYT::TNode>(spec, outputBuilder.GetTables());

        tx->Map(spec, new TMapper(outputBuilder.GetIndexes(), uploadTimestamp));
    }
}

TParseRawBindingsJob::TParseRawBindingsJob(const TConfig& config, NLog::TLogPtr log)
    : Config(config)
    , Log(log)
{
    Y_ENSURE(log != nullptr);
}

int TParseRawBindingsJob::Do() {
    auto client = NYT::CreateClient(Config.YtProxy);
    NYT::TConfig::Get()->Pool = Config.YtPool;

    for (const auto& srcTableNode : ListBindingsTables(client, Config.InputDir)) {
        TTimedSuffixGenerator timedSuffixGenerator;

        const auto& srcTableName = srcTableNode.AsString();
        const auto& srcTablePath = NYT::JoinYPaths(Config.InputDir, srcTableName);

        auto tx = client->StartTransaction();
        const auto& uploadAttrs = GetAttribute(tx, srcTablePath, UPLOAD_ATTRIBUTE_NAME);
        const auto& syncMode = FromString<ESyncMode>(uploadAttrs.At(NUploadAttrsFields::SYNC_MODE).AsString());
        const auto& uploadTimestamp = uploadAttrs.At(NUploadAttrsFields::TIMESTAMP).AsUint64();

        const auto& outTablesName = timedSuffixGenerator.AppendSuffix(TTimedSuffixGenerator::RemoveSuffix<TString>(srcTableName));
        auto outputTable = NYT::TRichYPath(NYT::JoinYPaths(Config.OutputDir, outTablesName))
                                          .OptimizeFor(NYT::EOptimizeForAttr::OF_SCAN_ATTR);
        const auto errorsTable = NYT::TRichYPath(NYT::JoinYPaths(Config.ErrorsDir, outTablesName))
                                                .OptimizeFor(NYT::EOptimizeForAttr::OF_SCAN_ATTR)
                                                .Schema(GetRawBindingsErrorsSchema());

        if (syncMode == ESyncMode::ITER) {
            RunMap<TParseRawIterBindingsMapper>(tx, srcTablePath, outputTable.Schema(GetParsedIterBindingsSchema()), errorsTable, uploadTimestamp);
        } else if (syncMode == ESyncMode::FULL) {
            RunMap<TParseFullIterBindingsMapper>(tx, srcTablePath, outputTable.Schema(GetUnsortedIdBindingsSchema()), errorsTable, uploadTimestamp);
        } else {
            Cerr << "Unknown sync mode: '" << syncMode << "'";
            return 1;
        }

        SetAttribute(tx, outputTable.Path_, UPLOAD_ATTRIBUTE_NAME, uploadAttrs);
        SetAttribute(tx, errorsTable.Path_, UPLOAD_ATTRIBUTE_NAME, uploadAttrs);

        if (Config.ShouldDropInput) {
            tx->Remove(srcTablePath);
        }
        tx->Commit();

        SetTtl(client, errorsTable.Path_, TDuration::Days(Config.ErrorsTtl), ESetTtlMode::RemoveIfEmpty);
    }

    return 0;
}

REGISTER_MAPPER(TParseRawIterBindingsMapper);
REGISTER_MAPPER(TParseFullIterBindingsMapper);
