#include "prepare_bindings_to_upload_errors_schema.h"
#include "prepare_bindings_to_upload_job.h"
#include "prepare_bindings_to_upload_reducer.h"

#include <crypta/dmp/common/data/bb_upload_state_schema.h>
#include <crypta/dmp/common/data/bindings_id_fields.h>
#include <crypta/dmp/common/data/collector_schema.h>
#include <crypta/dmp/common/data/segment.h>
#include <crypta/dmp/common/helpers/meta.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <util/generic/hash_set.h>

using namespace NCrypta;
using namespace NDmp;
using namespace NAdobe;

TPrepareBindingsToUploadJob::TPrepareBindingsToUploadJob(const TConfig& config, NLog::TLogPtr log)
    : Config(config)
    , Log(log)
{
    Y_ENSURE(log != nullptr);
}

int TPrepareBindingsToUploadJob::Do() {
    Log->info("Do PrepareDataToBbJob");

    auto client = NYT::CreateClient(Config.YtProxy);
    NYT::TConfig::Get()->Pool = Config.YtPool;

    auto tx = client->StartTransaction();

    if (!tx->Exists(Config.YandexuidBindingsTable)) {
        Log->error("Yandexuid bindings table {} not exists", Config.YandexuidBindingsTable);
        return 1;
    }

    if (!tx->Exists(Config.MetaTable)) {
        Log->error("Meta table {} not exists", Config.MetaTable);
        return 1;
    }

    tx->Create(Config.BbUploadStateTable, NYT::NT_TABLE, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true).Attributes(NYT::TNode()("schema", GetBbUploadStateSchema().ToNode())));

    const auto& now = TShiftedClock::Now();
    TTimedYtPathGenerator timedPathGenerator(now);

    TPrepareBindingsToUploadReducer::TOutputIndexes::TBuilder outputBuilder;
    outputBuilder.Add(NYT::TRichYPath(Config.BbUploadStateTable).Schema(GetBbUploadStateSchema()),
                      TPrepareBindingsToUploadReducer::EOutputTables::BbUploadState);

    const auto& errorsPath = timedPathGenerator.GetPath(Config.ErrorsDir);
    outputBuilder.Add(NYT::TRichYPath(errorsPath).Schema(GetPrepareDataToBbErrorsSchema()),
                      TPrepareBindingsToUploadReducer::EOutputTables::Errors);

    if (Config.PrepareDataToBb) {
        const TString tableNameFormat = ToString(Config.DmpId) + "_{ts}";

        outputBuilder.Add(NYT::TRichYPath(timedPathGenerator.GetPath(Config.ToBbCollectorDir, tableNameFormat)).Schema(GetCollectorSchema()),
                          TPrepareBindingsToUploadReducer::EOutputTables::ToCollector);
    }

    auto spec = NYT::TReduceOperationSpec().ReduceBy(NBindingsIdFields::YANDEXUID);
    spec.SetInput<NYT::TNode>(TPrepareBindingsToUploadReducer::YANDEXUID_BINDINGS_INDEX, Config.YandexuidBindingsTable);
    spec.SetInput<NYT::TNode>(TPrepareBindingsToUploadReducer::BB_UPLOAD_STATE_INDEX, Config.BbUploadStateTable);
    AddOutputs<NYT::TNode>(spec, outputBuilder.GetTables());

    const auto& segmentsIds = NDmp::GetSegmentsIds(client, Config.MetaTable);

    tx->Reduce(spec, new TPrepareBindingsToUploadReducer(outputBuilder.GetIndexes(), segmentsIds, Config.UploadTtl, now.Seconds(), Config.DmpId));

    tx->Commit();

    SetTtl(client, errorsPath, TDuration::Days(Config.ErrorsTtl), ESetTtlMode::RemoveIfEmpty);

    Log->info("TPrepareBindingsToUploadJob Done");
    return 0;
}
