#include "prepare_bindings_to_upload_errors_fields.h"
#include "prepare_bindings_to_upload_reducer.h"

#include <crypta/dmp/common/serializers/bb_upload_state_serializer.h>
#include <crypta/dmp/common/serializers/bindings_serializer.h>
#include <crypta/dmp/common/serializers/collector_serializer.h>
#include <crypta/dmp/common/serializers/dmp_segments_proto_log_serializer.h>
#include <crypta/lib/native/yt/utils/helpers.h>

#include <util/string/builder.h>
#include <util/string/join.h>

using namespace NCrypta::NDmp;
using namespace NAdobe;

namespace {
    std::pair<TMaybe<TBindings>, TMaybe<TBbUploadState>> ReadInput(TPrepareBindingsToUploadReducer::TReader* reader) {
        TMaybe<TBindings> bindings;
        TMaybe<TBbUploadState> bbUploadState;

        if (reader->GetTableIndex() == TPrepareBindingsToUploadReducer::YANDEXUID_BINDINGS_INDEX) {
            bindings = NYandexuidBindingsSerializer::Deserialize(reader->GetRow());
            reader->Next();
        }

        if (reader->IsValid()) {
            bbUploadState = NBbUploadStateSerializer::Deserialize(reader->GetRow());
            reader->Next();
        }

        Y_ENSURE(!reader->IsValid());

        return {bindings, bbUploadState};
    }

    // TODO(kolontaev): Если похожий код есть в старом DMP, то надо вынести в общее место
    TVector<TSegmentId> RemoveUnknownSegments(TSegmentIds& segments, const THashSet<TSegmentId>& knownSegments) {
        TVector<TSegmentId> unknownSegments;
        for (auto it = segments.begin(); it != segments.end();) {
            if (!knownSegments.contains(*it)) {
                unknownSegments.push_back(*it);
                it = segments.erase(it);
            } else {
                ++it;
            }
        }
        return unknownSegments;
    }
}


TPrepareBindingsToUploadReducer::TPrepareBindingsToUploadReducer(const TOutputIndexes& outputIndexes, const THashSet<ui64>& segmentsIds, time_t uploadTtl, ui64 uploadTimestamp, ui64 dmpId)
    : OutputIndexes(outputIndexes)
    , SegmentsIds(segmentsIds)
    , UploadTtl(uploadTtl)
    , UploadTimestamp(uploadTimestamp)
    , DmpId(dmpId)
{
}

void TPrepareBindingsToUploadReducer::Do(TReader* reader, TWriter* writer) {
    auto [bindings, bbUploadState] = ReadInput(reader);

    if (!bindings.Defined()) {
        bbUploadState->Bindings.Segments.clear();
        WriteToCollector(writer, bbUploadState->Bindings);
        return;
    }

    auto validBindings = bindings.GetRef();
    const auto& unknownSegments = RemoveUnknownSegments(validBindings.Segments, SegmentsIds);
    if (!unknownSegments.empty()) {
        WriteErrors(writer, bindings.GetRef(), unknownSegments);
    }

    if (NeedToUpload(validBindings, bbUploadState)) {
        bbUploadState = TBbUploadState(std::move(validBindings), UploadTimestamp);
        WriteToCollector(writer, bbUploadState->Bindings);
    }

    if (bbUploadState.Defined()) {
        WriteBbUploadState(writer, bbUploadState.GetRef());
    }
}

void TPrepareBindingsToUploadReducer::WriteToCollector(TWriter* writer, const TBindings& bindings) const {
    if (OutputIndexes[EOutputTables::ToCollector].Defined()) {
        const auto& row = NCollectorSerializer::Serialize(NDmpSegmentsProtoLogSerializer::Serialize(bindings, DmpId));
        writer->AddRow(row, OutputIndexes[EOutputTables::ToCollector].GetRef());
    }
}

void TPrepareBindingsToUploadReducer::WriteBbUploadState(TWriter* writer, const TBbUploadState& bbUploadState) const {
    writer->AddRow(NBbUploadStateSerializer::Serialize(bbUploadState), OutputIndexes[EOutputTables::BbUploadState].GetRef());
}

void TPrepareBindingsToUploadReducer::WriteErrors(TWriter* writer, const TBindings& bindings, const TVector<TSegmentId>& unknownSegments) const {
    using namespace NPrepareDataToBbErrorsFields;

    const auto error = TStringBuilder() << "unknown segment" << (unknownSegments.size() == 1 ? " " : "s ") << JoinSeq(", ", unknownSegments);
    const auto& row = NYT::TNode()(YANDEXUID, bindings.Id)
                                  (BINDINGS, NYandexuidBindingsSerializer::Serialize(bindings))
                                  (ERROR, error);
    writer->AddRow(row, OutputIndexes[EOutputTables::Errors].GetRef());
}

bool TPrepareBindingsToUploadReducer::NeedToUpload(const TBindings& bindings, const TMaybe<TBbUploadState>& bbUploadState) const {
    return (!bbUploadState.Defined() && !bindings.Segments.empty()) ||
           (bbUploadState.Defined() && ((bindings.Segments != bbUploadState->Bindings.Segments) ||
                                        (UploadTtl.IsExpired(bbUploadState->LastUploadTimestamp))));
}

REGISTER_REDUCER(TPrepareBindingsToUploadReducer);
