import argparse
import datetime
import logging
import os

import yt.wrapper as yt
from boto.s3.connection import S3Connection

from crypta.dmp.adobe.bin.common.python import config_fields
from crypta.dmp.adobe.bin.transfer_bindings_to_yt.lib import transfer
from crypta.lib.python import yaml_config
from crypta.lib.python.logging import logging_helpers


logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser(description="Transfer bindings from S3 to YT")
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    config = args.config
    logger.info(config)

    s3_key = os.environ["CRYPTA_ADOBE_S3_KEY"]
    s3_secret = os.environ["CRYPTA_ADOBE_S3_SECRET"]
    bucket = S3Connection(s3_key, s3_secret, host=config[config_fields.S3_HOST]).get_bucket(config[config_fields.S3_BUCKET])

    yt.config.set_proxy(config[config_fields.YT_PROXY])
    yt.config["write_retries"]["enable"] = True
    yt.config["write_retries"]["chunk_size"] = 64 * 1024 * 1024
    yt.config["write_parallel"]["enable"] = True
    yt.config["write_parallel"]["unordered"] = True
    yt.config["write_parallel"]["max_thread_count"] = 10

    transfer.run(bucket,
                 yt,
                 destination_id=config[config_fields.DESTINATION_ID],
                 destination_dir=config[config_fields.RAW_EXT_ID_BINDINGS_DIR],
                 backup_dir=config[config_fields.S3_BACKUP_DIR],
                 backup_ttl=datetime.timedelta(days=config[config_fields.S3_BACKUP_TTL_DAYS]),
                 delete_s3_files=config[config_fields.S3_DELETE_FILES])
