import gzip
import json
import logging
import os
import shutil
import tempfile
import time

from yt import yson
import yt.wrapper as yt

from crypta.dmp.adobe.bin.common.python import bindings_filename
from crypta.lib.python import time_utils
from crypta.lib.python.yt import yt_helpers


GZIP_EXTENSION = ".gz"

logger = logging.getLogger(__name__)


def run(bucket, yt_client, destination_id, destination_dir, backup_dir, backup_ttl, delete_s3_files):
    yt.create("map_node", backup_dir, recursive=True, ignore_existing=True)

    keys = bucket.list()
    files_in_bucket = [key.name for key in keys]
    transfer_control_keys = sorted([key for key in keys if key.name.endswith(".info")], key=lambda x: x.name)
    failed = []

    for transfer_control_key in transfer_control_keys:
        try:
            metadata = bindings_filename.parse(transfer_control_key.name, extension_regexp=r"info")
        except Exception:
            continue

        if metadata.destination_id == destination_id:
            try:
                process(transfer_control_key, bucket, files_in_bucket, metadata, yt_client, destination_dir, backup_dir, backup_ttl, delete_s3_files)
            except Exception:
                logger.exception("Can't process")
                failed.append(transfer_control_key.name)
                continue

    if failed:
        raise Exception("Failed {}".format(failed))


def process(transfer_control_key, bucket, files_in_bucket, metadata, yt_client, destination_dir, backup_dir, backup_ttl, delete_s3_files):
    logger.info("Process %s", transfer_control_key.name)

    transfer_control_key_local_path = download_file(transfer_control_key.name, bucket)

    with open(transfer_control_key_local_path) as f:
        info = json.load(f)

    files = [fileinfo["FileName"] for fileinfo in sorted(info["Files"], key=lambda x: x["FileSequenceNumber"])]

    if any(filename not in files_in_bucket for filename in files):
        logger.info("Not enough files. Skip")
        return

    now = int(os.environ.get(time_utils.CRYPTA_FROZEN_TIME_ENV, time.time()))
    table = yt.ypath_join(destination_dir, "{}_{}".format(metadata.timestamp, now))

    yt_helpers.backup_local_file(transfer_control_key_local_path, yt.ypath_join(backup_dir, transfer_control_key.name), backup_ttl)

    with yt.Transaction():
        attributes = {
            "upload": {
                "sync_mode": metadata.sync_mode,
                "timestamp": yson.YsonUint64(metadata.timestamp)
            }
        }
        yt.create("table", table, recursive=True, ignore_existing=False, attributes=attributes)

        for filename in files:
            local_file_path = download_file(filename, bucket)
            yt_helpers.backup_local_file(local_file_path, yt.ypath_join(backup_dir, filename))

            if filename.endswith(GZIP_EXTENSION):
                local_file_path = ungzip(local_file_path)

            upload_to_yt(local_file_path, table)
            os.remove(local_file_path)

    for filename in files:
        yt_helpers.set_ttl(yt.ypath_join(backup_dir, filename), backup_ttl)

    if delete_s3_files:
        files_to_remove = files + [transfer_control_key.name]
        logger.info("Remove %s", files_to_remove)
        bucket.delete_keys(files_to_remove)

    logger.info("Done")


def download_file(filename, bucket):
    _, path = tempfile.mkstemp()
    key = bucket.get_key(filename)

    with open(path, "wb") as f:
        key.get_contents_to_file(f)

    return path


def ungzip(source_path):
    _, destination_path = tempfile.mkstemp()

    with open(destination_path, "wb") as output_stream, gzip.open(source_path, "rb") as input_stream:
        shutil.copyfileobj(input_stream, output_stream)

    os.remove(source_path)
    return destination_path


def upload_to_yt(filename, table):
    with open(filename) as f:
        bindings = ({"raw": line.strip("\n")} for line in f)
        yt.write_table(yt.TablePath(table, append=True), bindings, format=yt.JsonFormat(encoding="utf-8"), raw=False)
