import os
import time

import yt.wrapper as yt

from crypta.dmp.common.data.python import (
    meta,
    segment_status
)
from crypta.lib.python import time_utils


STATUS_MAP = {
    "ACTIVE": segment_status.ENABLED,
    "INACTIVE": segment_status.DISABLED,
    "DELETED": segment_status.DELETED
}


class DestinationMapping(object):
    def __init__(self, sid, name, description, status):
        self.sid = sid
        self.name = name
        self.description = description
        self.status = status

    def __hash__(self):
        return hash(tuple(self.__dict__.values()))

    def __eq__(self, other):
        return self.__dict__ == other.__dict__


def parse_destination_mapping(raw):
    return DestinationMapping(
        sid=raw["sid"],
        name=raw["elementName"],
        description=raw["elementDescription"],
        status=parse_status(raw["elementStatus"].upper())
    )


def parse_status(raw):
    status = STATUS_MAP.get(raw)
    if status is None:
        raise Exception("Unknown status {}".format(raw))
    return status


def create_segment(destination_mapping, hierarchy, tariff, acl, timestamp):
    return {
        "id": destination_mapping.sid,
        "title": {
            "en": destination_mapping.name
        },
        "hierarchy": {
            "en": hierarchy
        },
        "tariff": tariff,
        "description": {
            "en": destination_mapping.description
        },
        "acl": acl,
        "status": destination_mapping.status,
        "timestamp": timestamp
    }


def run(adobe_api_client, yt_client, destination_id, meta_table, hierarchy, tariff, acl):
    timestamp = int(os.environ.get(time_utils.CRYPTA_FROZEN_TIME_ENV, time.time()))
    unique_destination_mappings = set(parse_destination_mapping(x) for x in adobe_api_client.get_destination_mappings(destination_id))
    segments = sorted([create_segment(destination_mapping, hierarchy, tariff, acl, timestamp) for destination_mapping in unique_destination_mappings], key=lambda x: x["id"])
    with yt_client.Transaction():
        yt_client.create("table", meta_table, recursive=True, ignore_existing=True, attributes={"schema": meta.get_schema(), "optimize_for": "scan"})
        yt_client.write_table(meta_table, segments, format=yt.JsonFormat(encoding="utf-8"), raw=False)
