#include "update_bindings_job.h"
#include "update_full_bindings_reducer.h"
#include "update_iter_bindings_errors_schema.h"
#include "update_iter_bindings_reducer.h"

#include <crypta/dmp/adobe/bin/common/list_utils.h>
#include <crypta/dmp/adobe/bin/common/sync_mode.h>
#include <crypta/dmp/adobe/bin/common/upload_attrs_fields.h>
#include <crypta/dmp/adobe/bin/common/parsed_iter_bindings_schema.h>
#include <crypta/dmp/common/data/bindings_id_fields.h>
#include <crypta/dmp/common/data/bindings_schema.h>

#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_suffix_generator.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/util/ypath_join.h>

using namespace NCrypta;
using namespace NDmp;
using namespace NAdobe;

namespace {
    void SortById(NYT::IClientBasePtr client, const NYT::TRichYPath& srcTable, const NYT::TRichYPath& dstTable) {
        client->Sort(NYT::TSortOperationSpec().AddInput(srcTable)
                                              .Output(dstTable)
                                              .SortBy({NBindingsIdFields::ID}));
    }

    void RunUpdateIterBindingsReduce(NYT::IClientBasePtr client, const NYT::TRichYPath& diffTable, const NYT::TRichYPath& stateTable, const NYT::TRichYPath& errorsTable) {
        TUpdateIterBindingsReducer::TOutputIndexes::TBuilder outputBuilder;
        outputBuilder.Add(stateTable, TUpdateIterBindingsReducer::EOutputTables::BindingsState);
        outputBuilder.Add(errorsTable, TUpdateIterBindingsReducer::EOutputTables::Errors);

        auto spec = NYT::TReduceOperationSpec().ReduceBy({NBindingsIdFields::ID})
                                               .SetInput<NYT::TNode>(TUpdateIterBindingsReducer::STATE_TABLE_INDEX, stateTable)
                                               .SetInput<NYT::TNode>(TUpdateIterBindingsReducer::DIFF_TABLE_INDEX, diffTable);
        AddOutputs<NYT::TNode>(spec, outputBuilder.GetTables());

        client->Reduce(spec, new TUpdateIterBindingsReducer(outputBuilder.GetIndexes()));
    }

    void RunUpdateFullBindingsReduce(NYT::IClientBasePtr client, const NYT::TRichYPath& freshTable, const NYT::TRichYPath& stateTable) {
        auto spec = NYT::TReduceOperationSpec().ReduceBy({NBindingsIdFields::ID})
                                               .AddInput<NYT::TNode>(freshTable)
                                               .AddOutput<NYT::TNode>(stateTable);

        client->Reduce(spec, new TUpdateFullBindingsReducer());
    }
}

TUpdateBindingsJob::TUpdateBindingsJob(const TConfig& config, NLog::TLogPtr log)
    : Config(config)
    , Log(log)
{
    Y_ENSURE(log != nullptr);
}

int TUpdateBindingsJob::Do() {
    auto client = NYT::CreateClient(Config.YtProxy);
    NYT::TConfig::Get()->Pool = Config.YtPool;

    const auto dstTable = NYT::TRichYPath(Config.StateTable).Schema(GetIdBindingsSchema())
                                                            .OptimizeFor(NYT::EOptimizeForAttr::OF_SCAN_ATTR);

    client->Create(dstTable.Path_, NYT::ENodeType::NT_TABLE, NYT::TCreateOptions().IgnoreExisting(true)
                                                                                  .Attributes(NYT::TNode()("schema", GetIdBindingsSchema().ToNode())
                                                                                                          (LAST_UPLOAD_TIMESTAMP_ATTRIBUTE_NAME, 0u)));

    for (const auto& srcTableNode : ListBindingsTables(client, Config.InputDir)) {
        TTimedSuffixGenerator timedSuffixGenerator;
        const auto& srcTableName = srcTableNode.AsString();

        NYT::TYPath errorsTablePath = "";
        const auto& srcTablePath = NYT::JoinYPaths(Config.InputDir, srcTableName);

        Log->info("Processing table {}", srcTablePath);
        auto tx = client->StartTransaction();

        const auto& lastUploadTimestamp = GetAttribute(tx, dstTable.Path_, LAST_UPLOAD_TIMESTAMP_ATTRIBUTE_NAME).AsUint64();
        Log->info("Last processed upload timestamp (state table '{}' attribute) = {}", LAST_UPLOAD_TIMESTAMP_ATTRIBUTE_NAME, lastUploadTimestamp);

        const auto& uploadAttrs = GetAttribute(tx, srcTablePath, UPLOAD_ATTRIBUTE_NAME);
        const auto& syncMode = FromString<ESyncMode>(uploadAttrs.At(NUploadAttrsFields::SYNC_MODE).AsString());
        const auto& uploadTimestamp = uploadAttrs.At(NUploadAttrsFields::TIMESTAMP).AsUint64();
        Log->info("Src table attrs: path={}, sync_mode={}, upload_timestamp={}", srcTablePath, ToString(syncMode), uploadTimestamp);

        if (uploadTimestamp >= lastUploadTimestamp) {
            NYT::TTempTable sortedSrcTmpTable(tx, "sorted_src");
            SortById(tx, srcTablePath, sortedSrcTmpTable.Name());

            if (syncMode == ESyncMode::ITER) {
                errorsTablePath = NYT::JoinYPaths(Config.ErrorsDir, timedSuffixGenerator.AppendSuffix(TTimedSuffixGenerator::RemoveSuffix<TString>(srcTableName)));
                const auto errorsTable = NYT::TRichYPath(errorsTablePath).Schema(GetUpdateIterBindingsErrorsSchema())
                                                                         .OptimizeFor(NYT::EOptimizeForAttr::OF_SCAN_ATTR);

                RunUpdateIterBindingsReduce(tx, sortedSrcTmpTable.Name(), dstTable, errorsTable);
                SetAttribute(tx, errorsTable.Path_, UPLOAD_ATTRIBUTE_NAME, uploadAttrs);
            } else if (syncMode == ESyncMode::FULL) {
                RunUpdateFullBindingsReduce(tx, sortedSrcTmpTable.Name(), dstTable);
            } else {
                Log->error("Unknown sync mode: {}", ToString(syncMode));
                return 1;
            }
            SetAttribute(tx, dstTable.Path_, LAST_UPLOAD_TIMESTAMP_ATTRIBUTE_NAME, uploadTimestamp);
        } else {
            Log->info("Skipping table {}. Last processed upload timestamp () > src table upload timestamp ()", lastUploadTimestamp, uploadTimestamp);
        }

        if (Config.ShouldDropInput) {
            Log->info("Removing {}", srcTablePath);
            tx->Remove(srcTablePath);
        }
        tx->Commit();

        if (!errorsTablePath.empty()) {
            SetTtl(client, errorsTablePath, TDuration::Days(Config.ErrorsTtl), ESetTtlMode::RemoveIfEmpty);
        }
        Log->info("Table {} successfully processed", srcTablePath);
    }

    Log->info("All src tables processed successfully");
    return 0;
}
