#include "bindings_update_utils.h"
#include "update_iter_bindings_errors_fields.h"
#include "update_iter_bindings_reducer.h"

#include <crypta/dmp/adobe/bin/common/bindings_diff_serializer.h>
#include <crypta/dmp/common/serializers/bindings_serializer.h>

using namespace NCrypta::NDmp;
using namespace NAdobe;

namespace {
    using namespace NUpdateIterBindingsErrorsFields;

    NYT::TNode GetBaseErrorsNode(const TMaybe<TBindings>& state, const TMaybe<TBindingsDiff>& diff) {
        return NYT::TNode()(BINDINGS_STATE, state.Defined() ? NIdBindingsSerializer::Serialize(*state) : NYT::TNode::CreateEntity())
                           (BINDINGS_DIFF, diff.Defined() ? NBindingsDiffSerializer::Serialize(*diff) : NYT::TNode::CreateEntity());
    }

    NYT::TNode GetError(ui64 segmentId, TString&& operation, TString&& error) {
        return NYT::TNode()(SEGMENT_ID, segmentId)
                           (OPERATION, std::move(operation))
                           (ERROR, std::move(error));
    }

    NYT::TNode GetUserErrors(const TString& userId, const NYT::TNode& base, NYT::TNode&& errors) {
        return NYT::TNode(base)(ERRORS, std::move(errors))
                               (EXT_ID, userId);
    }

    std::pair<TMaybe<TBindings>, TMaybe<TBindingsDiff>> GetInputData(TUpdateIterBindingsReducer::TReader* reader) {
        TMaybe<TBindings> state;
        TMaybe<TBindingsDiff> diff;

        if (reader->GetTableIndex() == TUpdateIterBindingsReducer::STATE_TABLE_INDEX) {
            state = NIdBindingsSerializer::Deserialize(reader->GetRow());
            reader->Next();
        }

        if (reader->IsValid()) {
            diff = NBindingsDiffSerializer::Deserialize(reader->GetRow());
            reader->Next();

            for (; reader->IsValid(); reader->Next()) {
                const auto& additionalDiff = NBindingsDiffSerializer::Deserialize(reader->GetRow());
                const auto& addedSegments = additionalDiff.AddedSegments;
                const auto& removedSegments = additionalDiff.RemovedSegments;

                diff->AddedSegments.insert(addedSegments.begin(), addedSegments.end());
                diff->RemovedSegments.insert(removedSegments.begin(), removedSegments.end());
            }
        }

        return {state, diff};
    };
}

TUpdateIterBindingsReducer::TUpdateIterBindingsReducer(const TOutputIndexes& outputIndexes)
    : OutputIndexes(outputIndexes)
{}

void TUpdateIterBindingsReducer::Do(TReader* reader, TWriter* writer) {
    auto [state, diff] = GetInputData(reader);
    const auto& baseErrorsNode = GetBaseErrorsNode(state, diff);
    auto stateBindings = state.Defined() ? std::move(*state) : TBindings(diff->Id, diff->Timestamp, {});

    if (diff.Defined() && diff->Timestamp >= stateBindings.Timestamp) {
        auto errors = NYT::TNode::CreateList();

        stateBindings.Timestamp = diff->Timestamp;

        const auto& alreadyInStateSegments = NBindingsUpdateUtils::AddSegments(stateBindings.Segments, diff->AddedSegments);
        for (const auto& id: alreadyInStateSegments) {
            errors.Add(GetError(id, "add", "already in state"));
        }

        const auto& notFoundInStateSegments = NBindingsUpdateUtils::RemoveSegments(stateBindings.Segments, diff->RemovedSegments);
        for (const auto& id: notFoundInStateSegments) {
            errors.Add(GetError(id, "remove", "not found in state"));
        }

        if (!errors.Empty()) {
            writer->AddRow(GetUserErrors(stateBindings.Id, baseErrorsNode, std::move(errors)),
                           OutputIndexes[EOutputTables::Errors].GetRef());
        }
    }

    if (!stateBindings.Segments.empty()) {
        writer->AddRow(NIdBindingsSerializer::Serialize(stateBindings), OutputIndexes[EOutputTables::BindingsState].GetRef());
    }
}

REGISTER_REDUCER(TUpdateIterBindingsReducer);
