# -*- coding: utf-8 -*-

import argparse
import datetime
import logging

import yt.wrapper as yt

from crypta.dmp.adobe.bin.common.python import config_fields
from crypta.dmp.common.upload_to_audience import (
    flatten_segments_query,
    upload
)
from crypta.lib.python import yaml_config
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.yql import executer
from crypta.lib.python.yt import (
    path_utils,
    yt_helpers,
)

logger = logging.getLogger(__name__)


QUERY_TEMPLATE = """
INSERT INTO `{dst_table}` WITH TRUNCATE
{query}
ORDER BY id
"""


def parse_args():
    parser = argparse.ArgumentParser(description="Uploads AAM segments to Yandex.Audience")
    parser.add_argument("--config", help="Config file path", required=True, type=yaml_config.load)
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    logger.info("args: %s", args)

    yt_proxy = args.config[config_fields.YT_PROXY]
    yt_pool = args.config[config_fields.YT_POOL]
    yt_tmp_dir = args.config[config_fields.YT_TMP_DIR]

    yt.config.set_proxy(yt_proxy)
    yt.config["pool"] = yt_pool

    with yt.Transaction() as tx, yt.TempTable(yt_tmp_dir) as flattened_segments_table:
        query = QUERY_TEMPLATE.format(dst_table=flattened_segments_table,
                                      query=flatten_segments_query.QUERY_TEMPLATE.format(bindings_table=args.config[config_fields.FILTERED_YANDEXUID_BINDINGS_TABLE]))

        logger.info("Running YQL query:\n%s", query)
        executer.get_executer(yt_proxy, yt_pool, yt_tmp_dir)(query, transaction=tx.transaction_id, syntax_version=1)

        errors_table_path = path_utils.get_ts_table_path(args.config[config_fields.UPLOAD_TO_AUDIENCE_ERRORS_DIR])
        upload.upload_flattened_bindings(flattened_segments_table,
                                         args.config[config_fields.META_TABLE],
                                         args.config[config_fields.AUDIENCE_LOGIN],
                                         errors_table_path,
                                         args.config[config_fields.UPLOAD_TO_AUDIENCE_MEMORY_LIMIT],
                                         args.config[config_fields.AUDIENCE_SRC_TVM_ID],
                                         args.config[config_fields.AUDIENCE_API_DST_TVM_ID],
                                         max_concurrent_jobs=args.config[config_fields.UPLOAD_TO_AUDIENCE_MAX_CONCURRENT_JOBS],
                                         min_segment_size=args.config[config_fields.AUDIENCE_MIN_SEGMENT_SIZE])

    yt_helpers.set_ttl(errors_table_path, datetime.timedelta(days=args.config[config_fields.ERRORS_TTL_DAYS]))
    logger.info("Completed successfully")
