import argparse
import datetime
import logging

from boto.s3.connection import S3Connection

from crypta.dmp.adobe.monitorings.check_s3_bucket.lib import watcher
from crypta.dmp.adobe.monitorings.check_s3_bucket.lib.config_pb2 import TConfig
from crypta.lib.python import yaml_config
from crypta.lib.python.juggler.juggler_queue import JugglerQueue
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.proto_secrets import proto_secrets
from crypta.lib.python.solomon import reporter


logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", required=True, type=lambda x: yaml_config.parse_config(TConfig, x))
    parser.add_argument("--index", required=True, type=yaml_config.load)
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    config = args.config
    index = args.index

    logger.info("Config:\n%s", proto_secrets.get_copy_without_secrets(config))
    logger.info("Index:\n%s", index)

    bucket = S3Connection(config.S3.Key, config.S3.Secret, host=config.S3.Host).get_bucket(config.S3.Bucket)
    bucket_status = watcher.get_bucket_status(keys=bucket.list(), destination_ids=[item["destination_id"] for item in index])

    send_metrics(config, bucket_status)
    send_events(config, bucket_status)

    logger.info("Finish")


def send_metrics(config, bucket_status):
    if not config.DryRun:
        solomon_reporter = reporter.create_solomon_reporter_from_config(config.Solomon)
        solomon_reporter.set_value("latency", bucket_status.latency)
        solomon_reporter.set_value("errors", len(bucket_status.errors))
        solomon_reporter.set_value("with_unknown_destinations", bucket_status.with_unknown_destinations)


def send_events(config, bucket_status):
    juggler = JugglerQueue(dry_run=config.DryRun)

    def send_event(service, crit, crit_msg):
        if crit:
            juggler.enqueue_crit(config.JugglerHost, service, crit_msg)
            logger.info(crit_msg)
        else:
            juggler.enqueue_ok(config.JugglerHost, service, "OK")

    send_event(
        "s3-bucket",
        bucket_status.errors,
        "Errors found while checking s3 bucket: {}".format("; ".join(bucket_status.errors)),
    )
    send_event(
        "s3-bucket-latency",
        bucket_status.latency > config.LatencyThresholdDays * 86400,
        "Too old files found. Latency = {}".format(datetime.timedelta(seconds=bucket_status.latency)),
    )
    send_event(
        "s3-bucket-with-unknown-destinations",
        bucket_status.with_unknown_destinations > config.WithUnknownDestinationsThreshold,
        "{} files with unknown destinations".format(bucket_status.with_unknown_destinations),
    )
