import datetime
import os

from library.python import resource
import yaml

from crypta.dmp.adobe.bin.common.python import config_fields
import crypta.dmp.common.check_upload_to_audience_errors.lib.check as upload_to_audience_check
from crypta.lib.python import templater
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import juggler_check_generator
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_latency_metric import YtLatencyMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from sandbox.projects.crypta.dmp.adobe import process_segments


def render_path(prefix, index_item, suffix):
    return os.path.join(prefix, suffix).format(**index_item)


def get_registry():
    index = yaml.safe_load(resource.find("/adobe_index"))
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-adobe"], host="crypta-adobe")

    prefix = "dmp/adobe/customers/{customer}/{destination_id}_{ext_id_type}"

    yt_size_dashboard_paths = [
        "ext_id_bindings",
        "yandexuid_bindings",
        "filtered_yandexuid_bindings",
        "errors/update_bindings",
        "errors/prepare_bindings_to_upload",
    ]

    monitored_error_paths = [
        ("errors/parse_bindings", DataSize(mb=0)),
        ("errors/upload_to_audience", DataSize(mb=1)),
    ]

    yt_latencies_paths = [
        ("parsed_ext_id_bindings", datetime.timedelta(hours=4)),
        ("raw_ext_id_bindings", datetime.timedelta(hours=4)),
    ]

    yt_registry = yt_config_registry.DeprecatedCryptaYtConfigRegistry(juggler)

    sandbox = sandbox_scheduler.create_default_generator(juggler, ["ADOBE"])

    for item in index:
        for path in yt_size_dashboard_paths:
            YtSizeMetric(yt_registry, render_path(prefix, item, path))

        for path, threshold in yt_latencies_paths:
            YtLatencyMetric(yt_registry, render_path(prefix, item, path)).add_latency_alert(
                threshold=threshold,
            )

        for path, threshold in monitored_error_paths:
            YtSizeMetric(yt_registry, render_path(prefix, item, path)).add_disk_space_alert(
                predicate=alert_pb2.GT,
                threshold=threshold,
            )

        for env, kill_timeout in (
            (environment.STABLE, datetime.timedelta(hours=12)),
            (environment.TESTING, datetime.timedelta(minutes=70)),
        ):
            scheduler = sandbox.create_scheduler(
                process_segments.CryptaAdobeProcessSegmentsTask,
                schedule_interval=datetime.timedelta(hours=4),
                kill_timeout=kill_timeout,
                extra_params={key: item[key] for key in ["customer", "ext_id_type"]},
                env=env,
                requirements={"dns": "dns64"},
            )
            if env == environment.STABLE:
                scheduler.check(
                    datetime.timedelta(hours=24),
                    juggler_service=process_segments.CryptaAdobeProcessSegmentsTask.get_juggler_service_for_customer(item["customer"], item["ext_id_type"]),
                ).add_yt_dependencies(
                    YtProxy.Group.offline,
                )

        config = yaml.safe_load(templater.render_resource("/adobe_config", vars=dict(environment=environment.STABLE, **item), strict=True))
        upload_to_audience_check.add(juggler, config[config_fields.UPLOAD_TO_AUDIENCE_ERRORS_DIR], item["customer"])

    for service in ["s3-bucket", "s3-bucket-latency", "s3-bucket-with-unknown-destinations"]:
        juggler.direct(service)

    return juggler
