import logging

import yt.wrapper as yt

from crypta.dmp.common.data.python import meta
from crypta.lib.python import templater
from crypta.lib.python.yql import executer
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)

QUERY_TEMPLATE = """
PRAGMA SimpleColumns;

DEFINE SUBQUERY $flattened_segment_ids($bindings_table) AS
    SELECT
        segments AS segment_id
    FROM (
        SELECT
            Yson::ConvertToUint64List(segments) AS segments
        FROM $bindings_table
    )
    FLATTEN BY segments;
END DEFINE;

DEFINE SUBQUERY $sizes($bindings_table) AS
    SELECT
        segment_id,
        COUNT(*) as size
    FROM $flattened_segment_ids($bindings_table)
    GROUP BY segment_id;
END DEFINE;

INSERT INTO `{{meta_with_sizes_table}}` WITH TRUNCATE
SELECT
    meta.*,
    {% if ext_id_bindings_table is not none %} ext_id_sizes.size ?? {% endif %} 0ul AS ext_id_size,
    {% if yandexuid_bindings_table is not none %} yuid_sizes.size ?? {% endif %} 0ul AS yuid_size
FROM `{{meta_table}}` AS meta
{% if ext_id_bindings_table is not none %}
LEFT JOIN $sizes("{{ext_id_bindings_table}}") AS ext_id_sizes
ON meta.id == ext_id_sizes.segment_id
{% endif %}
{% if yandexuid_bindings_table is not none %}
LEFT JOIN $sizes("{{yandexuid_bindings_table}}") AS yuid_sizes
ON meta.id == yuid_sizes.segment_id
{% endif %}
;
"""


def run(yt_proxy,
        yt_pool,
        yt_tmp_dir,
        meta_table,
        meta_with_sizes_table,
        ext_id_bindings_table=None,
        yandexuid_bindings_table=None):

    yt_client = yt_helpers.get_yt_client(yt_proxy, yt_pool)
    yql_executer = executer.get_executer(yt_proxy, yt_pool, yt_tmp_dir)

    with yt_client.Transaction() as tx, yt_client.TempTable(yt_tmp_dir) as yql_query_dst_table:
        query = templater.render_template(QUERY_TEMPLATE, {
            "meta_table": meta_table,
            "meta_with_sizes_table": yql_query_dst_table,
            "ext_id_bindings_table": ext_id_bindings_table,
            "yandexuid_bindings_table": yandexuid_bindings_table
        })

        logger.info("Running YQL query:\n%s", query)
        yql_executer(query, transaction=tx.transaction_id, syntax_version=1)

        destination_table = yt.TablePath(meta_with_sizes_table, schema=meta.get_schema_with_sizes(), attributes={"optimize_for": "scan"})
        yt_client.create("map_node", yt.ypath_dirname(destination_table), recursive=True, ignore_existing=True)
        yt_client.run_sort(yql_query_dst_table, destination_table, sort_by="id")

    logger.info("Completed successfully")
