import argparse
import logging
import sys

import yt.wrapper as yt

from crypta.dmp.common.check_upload_to_audience_errors.lib.check import Check
from crypta.dmp.common.upload_to_audience import upload
from crypta.lib.python import yaml_config
from crypta.lib.python.juggler.juggler_client import JugglerClient
from crypta.lib.python.logging import logging_helpers
from crypta.spine import spine

logger = logging.getLogger(__name__)

CUSTOM_CHECKS_FIELD = "custom-checks"
MAX_SEGMENTS_IN_DESCRIPTION = 7


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def read_segment_ids(path):
    return set(row[upload.AAM_SEGMENT_ID_FIELD] for row in yt.read_table(yt.TablePath(path, columns=[upload.AAM_SEGMENT_ID_FIELD])))


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    config = args.config
    logging.info(config)

    yt.config.set_proxy(config["yt-proxy"])
    yt.config["pool"] = config["yt-pool"]

    juggler_client = JugglerClient(config["juggler-host"], config["juggler-port"], config["juggler-source"])

    if CUSTOM_CHECKS_FIELD in config:
        logger.info("Use custom checks from config")
        checks = [Check(**kwargs) for kwargs in config[CUSTOM_CHECKS_FIELD]]
    else:
        checks = spine.get_config_registry().get_configs(Check.REGISTRY_TAG)

    exit_code = 0

    for check in checks:
        try:
            logger.info("Process %s", check)

            tables = yt.list(check.path, sort=True, absolute=True)

            if len(tables) > 1:
                penultimate_table_path, last_table_path = tables[-2:]
                penultimate_segment_ids = read_segment_ids(penultimate_table_path)
                last_segments_ids = read_segment_ids(last_table_path)
                not_uploaded_twice = list(last_segments_ids.intersection(penultimate_segment_ids))
            else:
                not_uploaded_twice = []

            if not_uploaded_twice:
                description = "These segments were not uploaded at least twice: {}{}".format(
                    ", ".join(str(i) for i in not_uploaded_twice[:MAX_SEGMENTS_IN_DESCRIPTION]),
                    "..." if len(not_uploaded_twice) > MAX_SEGMENTS_IN_DESCRIPTION else ""
                )
                juggler_client.send_warn(check.juggler_host, check.juggler_service, description)
            else:
                juggler_client.send_ok(check.juggler_host, check.juggler_service, "OK!")

        except Exception:
            logger.exception("")
            exit_code = 1

    sys.exit(exit_code)
