from datetime import timedelta
import os
import time

from crypta.dmp.common.metrics import common
from crypta.lib.python import (
    templater,
    time_utils,
)
from crypta.lib.python.yql import yql_helpers


DAYS_COUNT_TO_USE_ALL_DATA = 0


QUERY_TEMPLATE = """
SELECT
    COUNT(*)
FROM `{{table}}`
{% if min_timestamp != 0 %}
WHERE `timestamp` > {{min_timestamp}}
{% endif %}
;
"""


def get(yt_client, yql_executer, ext_id_bindings_table, yuid_bindings_table, days):
    if not yt_client.exists(ext_id_bindings_table) or not yt_client.exists(yuid_bindings_table):
        return {}

    ext_id_count, yuid_count = calc_id_counts(yt_client, yql_executer, ext_id_bindings_table, yuid_bindings_table, days)

    return get_metrics_dict(ext_id_count, yuid_count, days)


def calc_id_counts(yt_client, yql_executer, ext_id_bindings_table, yuid_bindings_table, days):
    current_timestamp = int(os.getenv(time_utils.CRYPTA_FROZEN_TIME_ENV, time.time()))
    min_timestamp = (current_timestamp - timedelta(days=days).total_seconds()) if days != DAYS_COUNT_TO_USE_ALL_DATA else 0

    queries = [
        templater.render_template(QUERY_TEMPLATE, dict(min_timestamp=min_timestamp, table=ext_id_bindings_table)),
        templater.render_template(QUERY_TEMPLATE, dict(min_timestamp=min_timestamp, table=yuid_bindings_table))
    ]

    with yt_client.Transaction() as tx:
        query_result = yql_executer("".join(queries), transaction=tx.transaction_id, syntax_version=1)
        ext_id_count = int(yql_helpers.get_single_value(query_result[0]))
        yuid_count = int(yql_helpers.get_single_value(query_result[1]))

    return ext_id_count, yuid_count


def get_metrics_dict(ext_id_count, yuid_count, days):
    time_interval = get_metric_time_node(days)
    return {
        'coverage.{}.ext_id'.format(time_interval): ext_id_count,
        'coverage.{}.yuid'.format(time_interval): yuid_count,
        'matching_rate.{}.user'.format(time_interval): common.calc_matching_rate(yuid_count, ext_id_count)
    }


def get_metric_time_node(days):
    return "{}d".format(days) if days != DAYS_COUNT_TO_USE_ALL_DATA else "total"
