import collections

from crypta.dmp.common.data.python import (
    segment_fields,
    segment_status
)
from crypta.dmp.common.metrics import common


class SegmentsSizesAccum:
    def __init__(self, count=0, yuid_size=0, ext_id_size=0):
        self.count = count
        self.ext_id_size = ext_id_size
        self.yuid_size = yuid_size


def calc_segments_rate(enabled, disabled):
    not_deleted = enabled + disabled
    return float(enabled) / not_deleted if not_deleted != 0 else 0


def calc(sizes_accum):
    enabled_segments = sizes_accum[segment_status.ENABLED]
    disabled_segments = sizes_accum[segment_status.DISABLED]
    deleted_segments = sizes_accum[segment_status.DELETED]

    return {
        'segments_count.enabled': enabled_segments.count,
        'segments_count.disabled': disabled_segments.count,
        'segments_count.deleted': deleted_segments.count,

        'segments_size.enabled.ext_id': enabled_segments.ext_id_size,
        'segments_size.enabled.yuid': enabled_segments.yuid_size,

        'segments_size.disabled.ext_id': disabled_segments.ext_id_size,
        'segments_size.disabled.yuid': disabled_segments.yuid_size,

        'segments_size.deleted.ext_id': deleted_segments.ext_id_size,
        'segments_size.deleted.yuid': deleted_segments.yuid_size,

        'matching_rate.total.bindings.enabled': common.calc_matching_rate(enabled_segments.yuid_size, enabled_segments.ext_id_size),
        'enabled_segments_rate': calc_segments_rate(enabled_segments.count, disabled_segments.count),

        'enabled_segments_size_rate.ext_id': calc_segments_rate(enabled_segments.ext_id_size, disabled_segments.ext_id_size),
        'enabled_segments_size_rate.yuid': calc_segments_rate(enabled_segments.yuid_size, disabled_segments.yuid_size)
    }


def read_segments_sizes(meta):
    sizes_accum = collections.defaultdict(SegmentsSizesAccum)

    for segment in meta:
        status = segment[segment_fields.STATUS]
        if not segment_status.is_status_valid(status):
            raise Exception('unknown status "{}"'.format(status))

        yuid_size = int(segment.get(segment_fields.YANDEXUID_SIZE, 0))
        ext_id_size = int(segment.get(segment_fields.EXT_ID_SIZE, 0))

        sizes_accum[status].count += 1
        sizes_accum[status].yuid_size += yuid_size
        sizes_accum[status].ext_id_size += ext_id_size

    return sizes_accum


def get(meta):
    return calc(read_segments_sizes(meta))


def get_small_segments(meta, threshold):
    result = []
    for segment in meta:
        status = segment[segment_fields.STATUS]
        yuid_size = int(segment.get(segment_fields.YANDEXUID_SIZE, 0))

        if status == segment_status.ENABLED and yuid_size < threshold:
            result.append(segment[segment_fields.ID])

    return result
