# -*- coding: utf-8 -*-
import json
import logging
import os
from StringIO import StringIO
import traceback

import requests
import tvmauth
from yt import yson
import yt.wrapper as yt

from crypta.dmp.common.data.python import segment_status
from crypta.dmp.common.upload_to_audience import segment_name
from crypta.lib.python import retryable_http_client
import crypta.lib.python.audience.client as audience
from crypta.lib.python.identifiers import identifiers as id_lib
from crypta.lib.python.logging import logging_helpers
import crypta.lib.python.tvm.helpers as tvm
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)

UPLOAD_AUDIENCE_TVM_ID_VAR = "UPLOAD_AUDIENCE_TVM_ID"
UPLOAD_AUDIENCE_TVM_SECRET_VAR = "UPLOAD_AUDIENCE_TVM_SECRET"
SHARE_AUDIENCE_OAUTH_TOKEN_VAR = "SHARE_AUDIENCE_OAUTH_TOKEN"
AUDIENCE_API_TVM_ID_VAR = "AUDIENCE_API_TVM_ID"

ID_FIELD = "id"
AAM_SEGMENT_ID_FIELD = "aam_segment_id"
ERROR_FIELD = "error"


class AudienceSegment():
    def __init__(self, id, name, status):
        self.id = id
        self.name = name
        self.status = status

    def __str__(self):
        return str(self.__dict__)

    __repr__ = __str__


def get_errors_schema():
    schema = yson.YsonList([
        {"name": AAM_SEGMENT_ID_FIELD, "type": "uint64", "required": True, "sort_order": "ascending"},
        {"name": ERROR_FIELD, "type": "string", "required": True}
    ])
    schema.attributes["strict"] = True
    schema.attributes["unique_keys"] = True
    return schema


def get_title(multilang_title):
    titles = {}
    for locale, title in multilang_title.iteritems():
        titles[locale.split("_")[0]] = title

    res = None
    for lang in ["ru", "en", "ua", "tr"]:
        res = res or titles.get(lang)

    if res is None:
        raise Exception("No title found in meta")

    return res


@yt.with_context
class AudienceUploadReducer(object):
    TRIES_COUNT = 6
    DELAY_IN_SECS = 1
    RETRY_JITTER = (10, 20)

    META_TABLE_INDEX = 0
    YANDEXUID_BINDINGS_TABLE_INDEX = 1

    class Data(object):
        def __init__(self):
            self.meta = None
            self.yandexuids = StringIO()
            self.yandexuids_count = 0

    def __init__(self, login, min_segment_size=0, max_segment_size=50000000, audience_url=audience.AUDIENCE_API_URL, audience_port=audience.PORT):
        self.upload_audience_client = None
        self.share_audience_client = None
        self.audience_segments = None
        self.login = login
        self.min_segment_size = min_segment_size
        self.max_segment_size = max_segment_size
        self.audience_url = audience_url
        self.audience_port = audience_port
        self.logger = logging_helpers.register_stderr_logger("audience_client")

    def start(self):
        tvm_dst_id = int(os.environ[yt_helpers.get_yt_secure_vault_env_var_for(AUDIENCE_API_TVM_ID_VAR)])

        localhost_port = os.environ.get(yt_helpers.get_yt_secure_vault_env_var_for(tvm.TVM_TEST_PORT_ENV_VAR))

        tvm_client = tvmauth.TvmClient(tvmauth.TvmApiClientSettings(
            self_tvm_id=int(os.environ[yt_helpers.get_yt_secure_vault_env_var_for(UPLOAD_AUDIENCE_TVM_ID_VAR)]),
            self_secret=os.environ[yt_helpers.get_yt_secure_vault_env_var_for(UPLOAD_AUDIENCE_TVM_SECRET_VAR)],
            dsts=[tvm_dst_id],
            localhost_port=int(localhost_port) if localhost_port is not None else None,
        ))

        retries_settings = audience.RetriesSettings(self.TRIES_COUNT, self.DELAY_IN_SECS, self.RETRY_JITTER)

        self.upload_audience_client = audience.PrivateApiAudienceClient(tvm_client=tvm_client,
                                                                        tvm_dst_id=tvm_dst_id,
                                                                        logger=self.logger,
                                                                        url=self.audience_url,
                                                                        port=self.audience_port,
                                                                        retries_settings=retries_settings,
                                                                        )

        self.share_audience_client = audience.PublicApiAudienceClient(oauth_token=os.environ[yt_helpers.get_yt_secure_vault_env_var_for(SHARE_AUDIENCE_OAUTH_TOKEN_VAR)],
                                                                      logger=self.logger,
                                                                      url=self.audience_url,
                                                                      port=self.audience_port,
                                                                      retries_settings=retries_settings,
                                                                      )

        self.audience_segments = self.get_audience_segments()

    def get_audience_segments(self):
        res = {}
        for segment in self.upload_audience_client.list_segments(self.login):
            audience_segment_name = segment_name.deserialize(segment["name"])
            res[audience_segment_name.aam_segment_id] = AudienceSegment(segment["id"],
                                                                        audience_segment_name,
                                                                        segment["status"])
        return res

    @staticmethod
    def get_error_row(aam_segment_id, error_message):
        return {AAM_SEGMENT_ID_FIELD: aam_segment_id, ERROR_FIELD: error_message}

    def modify_segment(self, audience_segment_id, yandexuids):
        allowed_error_messages = set([
            u"Данные сегмента не изменились",
            u"It is possible to modify data of processed segments only",
        ])

        # Dirty hack
        try:
            self.upload_audience_client.modify_segment_with_data(yandexuids, audience_segment_id, "replace", self.login, check_size=False)
        except retryable_http_client.RetryableHttpClientError as exc:
            json_text = json.loads(exc.text)
            if not (json_text.get("code") == requests.codes.bad_request and json_text.get("message") in allowed_error_messages):
                raise

    def read(self, rows, context, partner_segment_id):
        data = self.Data()

        for row in rows:
            table_index = context.table_index

            if table_index == self.META_TABLE_INDEX:
                if data.meta is not None:
                    raise Exception("Two meta rows found: '{}' and '{}'".format(row, data.meta))
                data.meta = row
            elif table_index == self.YANDEXUID_BINDINGS_TABLE_INDEX:
                if data.yandexuids_count > self.max_segment_size:
                    self.logger.warn("Segment %s is too big, taking only %s yandexuids", partner_segment_id, self.max_segment_size)
                    break

                if data.yandexuids_count != 0:
                    data.yandexuids.write("\n")
                data.yandexuids.write(row["yandexuid"])
                data.yandexuids_count += 1
            else:
                raise Exception("Unknown table index {}. Row = {}".format(table_index, row))
        return data

    @staticmethod
    def normalize_logins(logins):
        return set([id_lib.Login(login).normalize for login in logins])

    def process_grants(self, audience_segment_id, acl):
        allowed_error_messages = set([
            u"Такой пользователь не существует.",
            u"Разрешение этому пользователю уже выдано.",
            u"Нельзя указывать логин владельца.",
        ])

        meta_grants = AudienceUploadReducer.normalize_logins(acl)
        audience_grants = AudienceUploadReducer.normalize_logins(self.share_audience_client.list_grants_users(audience_segment_id))

        if audience_grants != meta_grants:
            self.logger.info("Update grants for segment = %s. Meta grants = %s, audience grants = %s", audience_segment_id, meta_grants, audience_grants)

            added_grants = meta_grants - audience_grants
            removed_grants = audience_grants - meta_grants

            errors = []

            self.share_audience_client.delete_grants(audience_segment_id, removed_grants)
            for grant in added_grants:
                try:
                    self.share_audience_client.add_grant(audience_segment_id, grant)
                except retryable_http_client.RetryableHttpClientError as exc:
                    json_text = json.loads(exc.text)
                    if json_text.get("code") != requests.codes.bad_request or json_text.get("message") not in allowed_error_messages:
                        errors.append("Failed to share segment to {}. Error = {}".format(grant, str(exc)))

            if errors:
                raise Exception("Grants errors: {}".format(errors))

    def __call__(self, key, rows, context):
        partner_segment_id = key[ID_FIELD]
        audience_segment_id = None

        try:
            data = self.read(rows, context, partner_segment_id)

            if data is not None:
                if data.meta is None:
                    raise Exception("No meta found for segment id = {}".format(partner_segment_id))

                meta_segment_status = data.meta["status"]
                if not segment_status.is_status_valid(meta_segment_status):
                    raise Exception("Unknown segment status: {}. Segment id = {}".format(meta_segment_status, partner_segment_id))

                meta_title = get_title(data.meta["title"])
                audience_segment = self.audience_segments.get(partner_segment_id)

                if audience_segment is not None:
                    audience_segment_id = audience_segment.id
                    if audience_segment.name.title != meta_title:
                        self.upload_audience_client.update_segment_name(audience_segment_id,
                                                                        segment_name.get_raw_audience_segment_name(partner_segment_id, meta_title),
                                                                        self.login)

                if meta_segment_status == segment_status.DELETED:
                    if audience_segment_id is not None:
                        self.upload_audience_client.delete_segment(audience_segment_id, self.login)
                elif meta_segment_status == segment_status.DISABLED or data.yandexuids_count < self.min_segment_size:
                    if audience_segment_id is not None:
                        self.share_audience_client.delete_all_segment_grants(audience_segment_id)
                elif meta_segment_status == segment_status.ENABLED:
                    if audience_segment_id is None:
                        response = self.upload_audience_client.upload_segment_from_data(
                            data.yandexuids.getvalue(),
                            segment_name.get_raw_audience_segment_name(partner_segment_id, meta_title),
                            content_type="yuid",
                            hashed=False,
                            ulogin=self.login,
                            check_size=False,
                        )
                        audience_segment_id = response["id"]
                    else:
                        self.modify_segment(audience_segment_id, data.yandexuids.getvalue())

                    self.process_grants(audience_segment_id, data.meta["acl"])

        except Exception:
            yield self.get_error_row(partner_segment_id,
                                     "Failed to process segment (partner_id={}, audience_id={}). Exception: {}".format(partner_segment_id,
                                                                                                                       audience_segment_id,
                                                                                                                       traceback.format_exc()))


def upload_flattened_bindings(
    flattened_segments_table,
    meta_table_path,
    audience_login,
    errors_table_path,
    memory_limit,
    audience_scr_tvm_id,
    audience_dst_tvm_id,
    max_concurrent_jobs=10,
    min_segment_size=0,
    max_segment_size=50000000,
    audience_url=audience.AUDIENCE_API_URL,
    audience_port=audience.PORT,
):
    yt.create("map_node", os.path.dirname(errors_table_path), ignore_existing=True, recursive=True)
    logger.info("Running upload reduce")

    reducer = AudienceUploadReducer(audience_login, min_segment_size=min_segment_size, max_segment_size=max_segment_size,
                                    audience_url=audience_url, audience_port=audience_port)
    yt.run_reduce(reducer,
                  source_table=[
                      # order is important
                      meta_table_path,
                      flattened_segments_table
                  ],
                  destination_table=yt.TablePath(errors_table_path, schema=get_errors_schema()),
                  reduce_by=[ID_FIELD],
                  spec={
                      "secure_vault": {
                          UPLOAD_AUDIENCE_TVM_ID_VAR: audience_scr_tvm_id,
                          AUDIENCE_API_TVM_ID_VAR: audience_dst_tvm_id,
                          UPLOAD_AUDIENCE_TVM_SECRET_VAR: os.environ[UPLOAD_AUDIENCE_TVM_SECRET_VAR],
                          SHARE_AUDIENCE_OAUTH_TOKEN_VAR: os.environ[SHARE_AUDIENCE_OAUTH_TOKEN_VAR],
                          tvm.TVM_TEST_PORT_ENV_VAR: tvm.get_tvm_test_port(),
                      },
                      "reducer": {
                          "memory_limit": memory_limit
                      },
                      "resource_limits": {
                          "user_slots": max_concurrent_jobs
                      },
                      "max_speculative_job_count_per_task": 0,
                  })
