import argparse
import logging
import os
import sys

import yt.wrapper as yt

from crypta.dmp.yandex.bin.common.python import config_fields
import crypta.dmp.yandex.bin.ftp_to_yt.lib as ftp_to_yt
from crypta.lib.python.ftp.client import ftp_entry
from crypta.lib.python.ftp.client.ftp_client import FtpClient
from crypta.lib.python import yaml_config
from crypta.lib.python.logging import logging_helpers


logger = logging.getLogger(__name__)


def is_hidden(filepath):
    name = os.path.basename(os.path.abspath(filepath))
    return name.startswith(".")


def parse_args():
    parser = argparse.ArgumentParser(description='Uploads DMP segments from FTP to YT')
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    logger.info(args)
    logger.info("================ Start ================")
    config = args.config

    yt.config.set_proxy(config[config_fields.YT_PROXY])
    yt.config["write_retries"]["enable"] = True
    yt.config["write_retries"]["chunk_size"] = 64 * 1024 * 1024
    yt.config["write_parallel"]["enable"] = True
    yt.config["write_parallel"]["unordered"] = True
    yt.config["write_parallel"]["max_thread_count"] = 10

    ftp_client = FtpClient(
        config[config_fields.FTP_HOST],
        config[config_fields.FTP_PORT],
        config[config_fields.FTP_USER],
        os.environ["CRYPTA_DMP_FTP_PASSWORD"],
        config[config_fields.FTP_DIR],
    )

    try:
        files_on_ftp = sorted((
            entry.name
            for entry in ftp_client.list_entries()
            if (entry.attrs[ftp_entry.Attrs.type] == ftp_entry.Types.file) and (not is_hidden(entry.name))
        ))
        logger.info("Files on FTP: %s", files_on_ftp)
    except Exception:
        logger.exception("Can't get list of files on FTP")
        sys.exit(1)

    failed_files = []

    for filename in files_on_ftp:
        try:
            ftp_to_yt.process_file(filename,
                                   ftp_client,
                                   delete_file_from_ftp=config[config_fields.FTP_DELETE_FILES],
                                   destination_yt_dir=config[config_fields.RAW_SEGMENTS_DIR],
                                   quarantine_dir=config[config_fields.QUARANTINE_DIR],
                                   backup_yt_dir=config[config_fields.FTP_BACKUP_DIR],
                                   backup_ttl_days=config[config_fields.FTP_BACKUP_TTL_DAYS])
        except Exception:
            logger.exception("Error while processing %s", filename)
            failed_files.append(filename)

    if not failed_files:
        exit_code = 0
    else:
        exit_code = 1

    logger.info("================ Finish ================")
    sys.exit(exit_code)
