import datetime
import logging
import os
import shutil
import tempfile
import time

import yt.wrapper as yt

from crypta.dmp.yandex.bin.ftp_to_yt.lib import (
    exceptions,
    helpers,
    uploading
)
from crypta.lib.python import time_utils
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)

TRANSACTION_TIMEOUT_MS = 600 * 1000


def process_file(filename, ftp_client, delete_file_from_ftp, destination_yt_dir, quarantine_dir, backup_yt_dir, backup_ttl_days):
    logger.info("Process %s", filename)

    downloads_dir = tempfile.mkdtemp()
    data_dir = tempfile.mkdtemp()

    try:
        with yt.Transaction(timeout=TRANSACTION_TIMEOUT_MS):
            try:
                local_file_path = os.path.join(downloads_dir, filename)
                ftp_client.download(filename, local_file_path)

                yt.mkdir(backup_yt_dir, recursive=True)
                backup_yt_file = yt.ypath_join(backup_yt_dir, filename)

                if yt.exists(backup_yt_file):
                    backup_yt_file += "_{}".format(int(os.environ.get(time_utils.CRYPTA_FROZEN_TIME_ENV, time.time())))

                yt_helpers.backup_local_file(local_file_path, backup_yt_file)

                archive_ts = helpers.get_ts(filename)
                helpers.extract(local_file_path, data_dir)
                helpers.validate_archive_dir(data_dir)

                meta_file = os.path.join(data_dir, helpers.META_FILE)
                bindings_file = os.path.join(data_dir, helpers.BINDINGS_FILE)
                uploading.upload_segments_to_yt(destination_yt_dir, archive_ts, meta_file, bindings_file)

                if delete_file_from_ftp:
                    ftp_client.delete(filename)

            except exceptions.DmpArchiveError as e:
                uploading.upload_errors_to_yt(quarantine_dir, filename, [e.message])
                if delete_file_from_ftp:
                    ftp_client.delete(filename)

        yt_helpers.set_ttl(backup_yt_file, datetime.timedelta(days=backup_ttl_days))

    finally:
        logger.info("Remove tmp dirs %s", [downloads_dir, data_dir])
        shutil.rmtree(downloads_dir)
        shutil.rmtree(data_dir)
