import io
import json
import logging
import os

import yt.wrapper as yt

from crypta.dmp.yandex.bin.ftp_to_yt.lib import exceptions
from crypta.dmp.yandex.bin.common.python import (
    error_fields,
    errors_schema
)


logger = logging.getLogger(__name__)


def read_meta(filename):
    if not os.path.isfile(filename):
        raise exceptions.DmpArchiveError("meta file is missing")

    try:
        with io.open(filename, "r", encoding="utf-8-sig") as f:
            meta = json.load(f)
    except Exception:
        raise exceptions.DmpArchiveError("meta is not valid JSON")

    try:
        assert isinstance(meta, list) and all(isinstance(segment, dict) for segment in meta)
    except Exception:
        raise exceptions.DmpArchiveError("meta is not valid JSON list of dicts")

    if not meta:
        raise exceptions.DmpArchiveError("meta is empty")

    return meta


def upload_meta_to_yt(meta, table):
    rows = ({"raw": raw} for raw in meta)
    yt.write_table(table, rows, format=yt.JsonFormat(encoding="utf-8"), raw=False)


def upload_bindings_to_yt(filename, table):
    with open(filename) as f:
        bindings = ({"raw": line.strip("\r\n")} for line in f)
        yt.write_table(table, bindings, format=yt.JsonFormat(encoding="utf-8"), raw=False)


def upload_segments_to_yt(yt_dir, ts, meta_file, bindings_file):
    ts_yt_dir = yt.ypath_join(yt_dir, str(ts))
    meta_yt_table = yt.ypath_join(ts_yt_dir, "meta")
    bindings_yt_table = yt.ypath_join(ts_yt_dir, "bindings")

    logger.info("Read meta from %s ...", meta_file)
    meta = read_meta(meta_file)

    yt.mkdir(ts_yt_dir, recursive=True)

    logger.info("Uploading meta to %s ...", meta_yt_table)
    upload_meta_to_yt(meta, meta_yt_table)
    logger.info("Done")

    # bindings file is optional
    if os.path.isfile(bindings_file):
        logger.info("Uploading %s to %s ...", bindings_file, bindings_yt_table)
        upload_bindings_to_yt(bindings_file, bindings_yt_table)
        logger.info("Done")
    else:
        logger.info("No bindings file found")


def upload_errors_to_yt(quarantine_dir, filename, errors):
    row = {
        error_fields.SOURCE: "archive",
        error_fields.VALUE: None,
        error_fields.MESSAGES: errors
    }
    destination_dir = yt.ypath_join(quarantine_dir, filename)
    errors_table = yt.ypath_join(destination_dir, "errors")
    logger.info("Uploading processing error to %s", errors_table)
    yt.mkdir(destination_dir, recursive=True)
    yt.write_table(yt.TablePath(errors_table, schema=errors_schema.get(), attributes={"optimize_for": "scan"}), [row], raw=False)
    logger.info("Done")
