#include "parse_bindings_mapper.h"
#include "raw_bindings_parser.h"

#include <crypta/dmp/common/data/bindings_fields.h>
#include <crypta/dmp/common/serializers/bindings_serializer.h>
#include <crypta/dmp/yandex/bin/common/helpers.h>

#include <util/string/builder.h>
#include <util/string/join.h>

using namespace NCrypta::NDmp;

TParseBindingsMapper::TParseBindingsMapper(const TOutputIndexes& outputIndexes, ui64 timestamp, const THashSet<ui64>& segmentsIds)
    : OutputIndexes(outputIndexes)
    , Timestamp(timestamp)
    , SegmentsIds(segmentsIds)
{
}

void TParseBindingsMapper::Do(TReader* reader, TWriter* writer) {
    for (; reader->IsValid(); reader->Next()) {
        const auto& raw = reader->GetRow().At("raw").AsString();
        NYT::TNode row;
        EOutputTables outputTable;
        try {
            const TBindings bindings = ParseRawBindings(raw, Timestamp);
            TVector<ui64> unknownSegments;
            for (const auto& segment : bindings.Segments) {
                if (!SegmentsIds.contains(segment)) {
                    unknownSegments.push_back(segment);
                }
            }
            if (!unknownSegments.empty()) {
                TString msg = TStringBuilder() << (unknownSegments.size() == 1 ? "segment " : "segments ")
                                               << JoinSeq(",", unknownSegments)
                                               << " not found in meta";
                row = CreateError("bindings", NExtIdBindingsSerializer::Serialize(bindings), msg);
                outputTable = EOutputTables::Errors;
            } else {
                row = NExtIdBindingsSerializer::Serialize(bindings);
                outputTable = EOutputTables::Bindings;
            }
        } catch (const yexception&) {
            row = CreateError("bindings", raw, "invalid binding format");
            outputTable = EOutputTables::Errors;
        }
        writer->AddRow(row, OutputIndexes[outputTable].GetRef());
    }
}

REGISTER_MAPPER(TParseBindingsMapper);
