#include "raw_segment_validator.h"

#include <crypta/dmp/common/data/segment_fields.h>

#include <util/generic/algorithm.h>
#include <util/generic/map.h>

#include <regex>

namespace NCrypta::NDmp {

namespace {
    bool IsListOfNonemptyStrings(const NYT::TNode& node) {
        return node.IsList() && AllOf(node.AsList(), [](const NYT::TNode& value) {
            return value.IsString() && !value.Empty();
        });
    }

    bool IsLocale(const TString& s) {
        static const auto localePattern = std::regex("^(ru|en|ua|tr)(_..)?$", std::regex_constants::icase);
        return std::regex_search(s.begin(), s.end(), localePattern);
    }

    bool IsInternatiolizedString(const NYT::TNode& node, size_t maxSize) {
        return node.IsMap() && !node.Empty() && AllOf(node.AsMap(), [maxSize](const NYT::TNode::TMapType::value_type& kv) {
            return IsLocale(kv.first) && kv.second.IsString() && !kv.second.Empty() && TUtf16String::FromUtf8(kv.second.AsString()).size() <= maxSize;
        });
    }

    bool IsId(const NYT::TNode& node) {
        return node.IsInt64() && node.AsInt64() >= 1 && node.AsInt64() <= 1000000000;
    }

    bool IsTitle(const NYT::TNode& node) {
        return IsInternatiolizedString(node, 100);
    }

    bool IsHierarchy(const NYT::TNode& node) {
        return node.IsMap() && !node.Empty() && AllOf(node.AsMap(), [](const NYT::TNode::TMapType::value_type& kv) {
            return IsLocale(kv.first) && IsListOfNonemptyStrings(kv.second) && kv.second.AsList().size() <= 7;
        });
    }

    bool IsTariff(const NYT::TNode& node) {
        return node.IsInt64() && node.AsInt64() >= 1 & node.AsInt64() <= 5;
    }

    bool IsDescription(const NYT::TNode& node) {
        return IsInternatiolizedString(node, 200);
    }

    bool IsAcl(const NYT::TNode& node) {
        return IsListOfNonemptyStrings(node);
    }
}

NYT::TNode ValidateRawSegment(const NYT::TNode& node) {
    NYT::TNode errors = NYT::TNode::CreateList();

    TMap<TString, std::function<bool(const NYT::TNode&)>> requiredFields = {
        {NSegmentFields::ID, &IsId},
        {NSegmentFields::TITLE, &IsTitle},
        {NSegmentFields::HIERARCHY, &IsHierarchy},
        {NSegmentFields::TARIFF, &IsTariff},
    };

    TMap<TString, std::function<bool(const NYT::TNode&)>> optionalFields = {
        {NSegmentFields::DESCRIPTION, &IsDescription},
        {NSegmentFields::ACL, &IsAcl}
    };

    for (const auto& [field, value] : node.AsMap()) {
        if (!requiredFields.contains(field) && !optionalFields.contains(field)) {
            errors.Add("unknown field \"" + field + "\"");
        }
    }

    for (const auto& [field, check] : requiredFields) {
        if (!node.HasKey(field)) {
            errors.Add("required field \"" + field + "\" is missing");
        } else if (!check(node.At(field))) {
            errors.Add("invalid field \"" + field + "\"");
        }
    }

    for (const auto& [field, check] : optionalFields) {
        if (node.HasKey(field) && !check(node.At(field))) {
            errors.Add("invalid field \"" + field + "\"");
        }
    }

    return errors;
}

} // namespace NCrypta::NDmp
