#include "prepare_bindings_to_upload_job.h"
#include "prepare_bindings_to_upload_reducer.h"

#include <crypta/dmp/common/data/bb_upload_state_schema.h>
#include <crypta/dmp/common/data/bindings_id_fields.h>
#include <crypta/dmp/common/data/collector_schema.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

using namespace NCrypta;
using namespace NDmp;

TPrepareBindingsToUploadJob::TPrepareBindingsToUploadJob(const TConfig& config, NLog::TLogPtr log)
    : Config(config)
    , Log(log)
{
    Y_ENSURE(log != nullptr);
}

int TPrepareBindingsToUploadJob::Do() {
    Log->info("================ Start ================");

    auto client = NYT::CreateClient(Config.YtProxy);
    NYT::TConfig::Get()->Pool = Config.YtPool;

    auto tx = client->StartTransaction();

    const bool YandexuidBindingsTableExists = tx->Exists(Config.YandexuidBindingsTable);
    const bool BbUploadStateTableExists = tx->Exists(Config.BbUploadStateTable);

    if (!YandexuidBindingsTableExists && !BbUploadStateTableExists) {
        Log->info("Yandexuid bindings table {} and bb upload state table {} do not exist", Config.YandexuidBindingsTable, Config.BbUploadStateTable);
        return 0;
    }

    tx->Create(Config.BbUploadStateTable, NYT::NT_TABLE, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true).Attributes(NYT::TNode()("schema", GetBbUploadStateSchema().ToNode())));

    const auto& now = TShiftedClock::Now();
    TTimedYtPathGenerator timedPathGenerator(now);

    TPrepareBindingsToUploadReducer::TInputIndexes::TBuilder inputBuilder;
    if (YandexuidBindingsTableExists) {
        Log->info("Use yandexuid bindings table {}", Config.YandexuidBindingsTable);
        inputBuilder.Add(Config.YandexuidBindingsTable, TPrepareBindingsToUploadReducer::EInputTables::YandexuidBindings);
    }
    if (BbUploadStateTableExists) {
        Log->info("Use bb upload state table {}", Config.BbUploadStateTable);
        inputBuilder.Add(Config.BbUploadStateTable, TPrepareBindingsToUploadReducer::EInputTables::BbUploadState);
    }

    TPrepareBindingsToUploadReducer::TOutputIndexes::TBuilder outputBuilder;
    outputBuilder.Add(NYT::TRichYPath(Config.BbUploadStateTable).Schema(GetBbUploadStateSchema()),
                      TPrepareBindingsToUploadReducer::EOutputTables::BbUploadState);
    if (Config.BbUploadEnabled) {
        const TString tableNameFormat = ToString(Config.DmpId) + "_{ts}";

        const auto& toCollectorTable = timedPathGenerator.GetPath(Config.ToBbProtoDir, tableNameFormat);
        Log->info("Put proto bindings for uploading to {}", toCollectorTable);
        outputBuilder.Add(NYT::TRichYPath(toCollectorTable).Schema(GetCollectorSchema()),
                          TPrepareBindingsToUploadReducer::EOutputTables::ToCollector);
    } else {
        Log->info("Only update state");
    }

    auto spec = NYT::TReduceOperationSpec().ReduceBy(NBindingsIdFields::YANDEXUID);
    AddInputs<NYT::TNode>(spec, inputBuilder.GetTables());
    AddOutputs<NYT::TNode>(spec, outputBuilder.GetTables());

    tx->Reduce(spec, new TPrepareBindingsToUploadReducer(inputBuilder.GetIndexes(), outputBuilder.GetIndexes(), now.Seconds(), Config.DmpId, Config.UploadTtl));

    tx->Commit();

    Log->info("================ Finish ================");
    return 0;
}
