#include "prepare_bindings_to_upload_reducer.h"

#include <crypta/dmp/common/serializers/bb_upload_state_serializer.h>
#include <crypta/dmp/common/serializers/bindings_serializer.h>
#include <crypta/dmp/common/serializers/collector_serializer.h>
#include <crypta/dmp/common/serializers/dmp_segments_proto_log_serializer.h>
#include <crypta/lib/native/yt/utils/helpers.h>

using namespace NCrypta;
using namespace NDmp;

TPrepareBindingsToUploadReducer::TPrepareBindingsToUploadReducer(
    const TInputIndexes& inputIndexes,
    const TOutputIndexes& outputIndexes,
    ui64 uploadTimestamp,
    ui64 dmpId,
    time_t uploadTtl)
    : InputIndexes(inputIndexes)
    , OutputIndexes(outputIndexes)
    , UploadTimestamp(uploadTimestamp)
    , DmpId(dmpId)
    , UploadTtl(uploadTtl)
{
}

void TPrepareBindingsToUploadReducer::Do(TReader* reader, TWriter* writer) {
    auto [bindings, bbUploadState] = ReadInput(reader);

    if (!bindings.Defined()) {
        bbUploadState->Bindings.Segments.clear();
        WriteToCollector(writer, bbUploadState->Bindings);
        return;
    }

    if (NeedToUpload(bindings.GetRef(), bbUploadState)) {
        bbUploadState = TBbUploadState(std::move(bindings.GetRef()), UploadTimestamp);
        WriteToCollector(writer, bbUploadState->Bindings);
    }

    if (bbUploadState.Defined()) {
        WriteBbUploadState(writer, bbUploadState.GetRef());
    }
}

void TPrepareBindingsToUploadReducer::WriteBbUploadState(TWriter* writer, const TBbUploadState& bbUploadState) const {
    writer->AddRow(NBbUploadStateSerializer::Serialize(bbUploadState), OutputIndexes[EOutputTables::BbUploadState].GetRef());
}

void TPrepareBindingsToUploadReducer::WriteToCollector(TWriter* writer, const TBindings& bindings) const {
    if (OutputIndexes[EOutputTables::ToCollector].Defined()) {
        const auto& row = NCollectorSerializer::Serialize(NDmpSegmentsProtoLogSerializer::Serialize(bindings, DmpId));
        writer->AddRow(row, OutputIndexes[EOutputTables::ToCollector].GetRef());
    }
}

bool TPrepareBindingsToUploadReducer::NeedToUpload(const TBindings& bindings, const TMaybe<TBbUploadState>& bbUploadState) const {
    return (!bbUploadState.Defined() && !bindings.Segments.empty()) ||
           (bbUploadState.Defined() && ((bbUploadState->Bindings.Segments != bindings.Segments) ||
                                        UploadTtl.IsExpired(bbUploadState->LastUploadTimestamp)));
}

std::pair<TMaybe<TBindings>, TMaybe<TBbUploadState>> TPrepareBindingsToUploadReducer::ReadInput(TReader* reader) const {
    TMaybe<TBindings> bindings;
    TMaybe<TBbUploadState> bbUploadState;

    for (; reader->IsValid(); reader->Next()) {
        const auto tableIndex = reader->GetTableIndex();
        if (tableIndex == InputIndexes[EInputTables::YandexuidBindings]) {
            EnsureReadOnce<TBindings>(reader, bindings, NYandexuidBindingsSerializer::Deserialize);
        } else if (tableIndex == InputIndexes[EInputTables::BbUploadState]) {
            EnsureReadOnce<TBbUploadState>(reader, bbUploadState, NBbUploadStateSerializer::Deserialize);
        } else {
            ythrow yexception() << "Unknown table index " << tableIndex;
        }
    }

    return {bindings, bbUploadState};
}

REGISTER_REDUCER(TPrepareBindingsToUploadReducer);
