import argparse
import logging
import os

from crypta.dmp.common.metrics import (
    audience_metrics,
    coverage_metrics,
    meta_metrics
)
from crypta.dmp.yandex.bin.common.python import config_fields
from crypta.lib.python import yaml_config
import crypta.lib.python.audience.client as audience
from crypta.lib.python.graphite.sender import CryptaAPIGraphiteSender
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.solomon import reporter
from crypta.lib.python.yql import executer
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser(description="Gets meta and coverage metrics from YT and sends it to graphite")
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    logger.info(args)

    config = args.config
    yt_proxy = config[config_fields.YT_PROXY]
    yt_pool = config[config_fields.YT_POOL]

    yt_client = yt_helpers.get_yt_client(yt_proxy, yt_pool)
    yql_executer = executer.get_executer(yt_proxy, yt_pool, config[config_fields.YT_TMP_DIR])
    audience_client = audience.PublicApiAudienceClient(oauth_token=os.environ["CRYPTA_DMP_AUDIENCE_OAUTH_TOKEN"],
                                                       logger=logger,
                                                       url=config[config_fields.AUDIENCE_URL],
                                                       port=config[config_fields.AUDIENCE_PORT])
    graphite_sender = CryptaAPIGraphiteSender(token=os.environ["CRYPTA_API_TOKEN"],
                                              fqdn=config[config_fields.GRAPHITE_SOURCE_HOST],
                                              root_path="dmp.{}".format(config[config_fields.DMP_LOGIN]),
                                              dry_run=not config[config_fields.SEND_METRICS_TO_GRAPHITE])

    solomon_client = reporter.create_solomon_reporter(**config[config_fields.SOLOMON])
    metrics = meta_metrics.get(yt_client.read_table(config[config_fields.OUT_META_TABLE]))

    for days in config[config_fields.COVERAGE_METRICS_DAYS]:
        metrics.update(coverage_metrics.get(yt_client, yql_executer, config[config_fields.EXT_ID_BINDINGS_TABLE], config[config_fields.YANDEXUID_BINDINGS_TABLE], days=days))

    if config[config_fields.OBTAIN_AUDIENCE_METRICS]:
        metrics.update(audience_metrics.get(audience_client, config[config_fields.AUDIENCE_LOGIN], config[config_fields.UPLOAD_TO_AUDIENCE_ROBOT_NAME]))

    graphite_sender.send_metrics(metrics.iteritems())
    for sensor, value in metrics.iteritems():
        solomon_client.set_value(sensor, value, labels={"dmp": config[config_fields.DMP_LOGIN]})

    logger.info("Completed successfully")
