import argparse
import logging

from crypta.dmp.common.metrics import (
    coverage_metrics,
    meta_metrics
)
from crypta.dmp.yandex.bin.common.python import config_fields
from crypta.dmp.yandex.bin.send_metrics_mail.lib import message
from crypta.lib.python import yaml_config
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.smtp.text_mail_sender import TextMailSender
from crypta.lib.python.yql import executer
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser(description="Sends mail with DMP metrics")
    parser.add_argument("--config", help="Path to config", required=True, type=yaml_config.load)
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    logger.info(args)

    config = args.config
    yt_proxy = config[config_fields.YT_PROXY]
    yt_pool = config[config_fields.YT_POOL]

    yt_client = yt_helpers.get_yt_client(yt_proxy, yt_pool)
    yql_executer = executer.get_executer(yt_proxy, yt_pool, config[config_fields.YT_TMP_DIR])

    coverage_metrics_days = config[config_fields.COVERAGE_METRICS_DAYS_FOR_EMAILS]

    meta = list(yt_client.read_table(config[config_fields.OUT_META_TABLE]))
    metrics = meta_metrics.get(meta)
    metrics["small_segments"] = meta_metrics.get_small_segments(meta, threshold=message.SEGMENT_SIZE_THRESHOLD)
    for days in coverage_metrics_days:
        metrics.update(coverage_metrics.get(yt_client, yql_executer, config[config_fields.EXT_ID_BINDINGS_TABLE], config[config_fields.YANDEXUID_BINDINGS_TABLE], days))

    subject, body = message.get_stats_mail_content(config[config_fields.DMP_LOGIN], metrics, coverage_metrics_days)
    logger.info("Subject: %s", subject)
    logger.info("Body:\n%s", body)

    mail_sender = TextMailSender(
        host=config[config_fields.SMTP_HOST],
        port=config[config_fields.SMTP_PORT],
        default_from_addr=config[config_fields.SMTP_EMAIL_FROM],
        dry_run=not config[config_fields.SEND_METRICS_EMAILS]
    )

    mail_sender.send(subject, body,
                     to_addrs=config[config_fields.STATISTICS_EMAILS],
                     cc=config[config_fields.METRICS_EMAILS_CC],
                     bcc=config[config_fields.METRICS_EMAILS_BCC])

    if metrics["coverage.total.ext_id"] == 0:
        logger.info("Zero coverage found. Sending extra message")

        subject, body = message.get_errors_mail_content(config[config_fields.DMP_LOGIN])
        logger.info("Subject: %s", subject)
        logger.info("Body:\n%s", body)

        mail_sender.send(subject, body,
                         to_addrs=config[config_fields.ERRORS_EMAILS],
                         cc=config[config_fields.QUARANTINE_EMAILS_CC],
                         bcc=config[config_fields.QUARANTINE_EMAILS_BCC])

    logger.info("Completed successfully")
