from datetime import datetime
import os
import time

import pytz

from crypta.dmp.common.metrics import (
    coverage_metrics
)
from crypta.lib.python import time_utils


SEGMENT_SIZE_THRESHOLD = 1000

STATS_SUBJECT_TEMPLATE = "FTP user '{dmp_login}'. Summary statistics at {time}"
STATS_MESSAGE_TEMPLATE = """
Summary statistics at {time}

Segments: {segments_count:,}
Total cookies: {extids_count:,}
Matched cookies: {yuids_count:,}
Matching rate: {matching_rates}
{small_segments_warning}
"""

SMALL_SEGMENTS_WARNING_TEMPLATE = "Segments with less than {threshold} matched cookies (not available in Audience): {small_segments}\n"

ZERO_COVERAGE_SUBJECT_TEMPLATE = "FTP user '{dmp_login}'. Zero coverage at {time}"
ZERO_COVERAGE_BODY = """
WARNING: ALL YOUR SEGMENTS NOW CONTAIN ZERO IDENTIFIERS (USERS).

Possible causes are:
* no archives from you on our FTP (ftp.yandex.ru) for more than 14 days;
* archives were uploaded, but had some errors.

Please check that:
* the last upload to FTP took place less than 14 days ago;
* there were no email notifications about upload errors.
"""

MATCHING_RATE_TEMPLATE = "{matching_rate:.1%} ({time_period})"


def get_errors_mail_content(dmp_login):
    return ZERO_COVERAGE_SUBJECT_TEMPLATE.format(dmp_login=dmp_login, time=get_ts()), ZERO_COVERAGE_BODY


def get_stats_mail_content(dmp_login, metrics, coverage_days_list):
    ts = get_ts()
    return get_stats_subject(dmp_login, ts), get_stats_message_body(metrics, ts, coverage_days_list)


def get_ts():
    current_timestamp = int(os.getenv(time_utils.CRYPTA_FROZEN_TIME_ENV, time.time()))
    return datetime.utcfromtimestamp(current_timestamp).replace(tzinfo=pytz.timezone("UTC"))\
                                                       .astimezone(pytz.timezone("Europe/Moscow"))\
                                                       .strftime('%Y-%m-%d %H:%M:%S %Z')


def get_stats_subject(dmp_login, ts):
    return STATS_SUBJECT_TEMPLATE.format(dmp_login=dmp_login, time=ts)


def get_stats_message_body(metrics, ts, coverage_days_list):
    return STATS_MESSAGE_TEMPLATE.format(
        time=ts,
        segments_count=metrics["segments_count.enabled"],
        extids_count=metrics["coverage.total.ext_id"],
        yuids_count=metrics["coverage.total.yuid"],
        matching_rates=get_matching_rates_message(metrics, coverage_days_list),
        small_segments_warning=get_small_segments_warning(metrics),
    )


def get_small_segments_warning(metrics):
    small_segments = ", ".join([str(x) for x in metrics["small_segments"]])
    return SMALL_SEGMENTS_WARNING_TEMPLATE.format(threshold=SEGMENT_SIZE_THRESHOLD, small_segments=small_segments) if small_segments else ""


def get_matching_rates_message(metrics, coverage_days_list):
    messages = []

    for days in coverage_days_list:
        time_node = coverage_metrics.get_metric_time_node(days)
        matching_rate_metric_name = "matching_rate.{}.user".format(time_node)
        messages.append(MATCHING_RATE_TEMPLATE.format(
            matching_rate=metrics.get(matching_rate_metric_name, 0),
            time_period="total" if days == coverage_metrics.DAYS_COUNT_TO_USE_ALL_DATA else "last {}h".format(days * 24)
        ))

    return ", ".join(messages)
