import collections

from crypta.dmp.common.data.python import (
    bindings,
    segment_fields
)
from crypta.dmp.yandex.bin.common.python import error_fields
from crypta.lib.python import templater


SUBJECT_TEMPLATE = "Ftp user '{dmp_login}'. Errors in '{archive_name}'"
BODY_TEMPLATE = """
Processing of archive '{{archive_name}}' has failed.

{{description}}

The archive has been removed from FTP server.
Please, fix these problems and upload the archive again.
For more details, reply to this email.
"""
DESCRIPTION_TEMPLATE = """
Examples of {{source}} errors:
{% for error in errors %}
* {{error}}
{% endfor %}
"""

ARCHIVE = "archive"
META = "meta"
BINDINGS = "bindings"


def get_subject(archive_name, dmp_login):
    return SUBJECT_TEMPLATE.format(archive_name=archive_name, dmp_login=dmp_login)


def get_body(archive_name, rows):
    errors = collections.defaultdict(list)
    for row in rows:
        errors[row[error_fields.SOURCE]].append(format_error(row))

    descriptions = [templater.render_template(DESCRIPTION_TEMPLATE, {"source": source, "errors": errors[source]}).strip() for source in (ARCHIVE, META, BINDINGS) if errors[source]]

    return templater.render_template(BODY_TEMPLATE, {
        "archive_name": archive_name,
        "description": "\n".join(descriptions),
    })


def format_error(row):
    source = row[error_fields.SOURCE]
    value = row[error_fields.VALUE]
    messages = row[error_fields.MESSAGES]

    if source == ARCHIVE:
        return format_archive_error(messages)
    elif source == META:
        return format_meta_error(value, messages)
    elif source == BINDINGS:
        return format_bindings_error(value, messages)
    else:
        raise Exception("Unknown error source {}".format(source))


def format_archive_error(messages):
    return "; ".join(messages)


def format_meta_error(value, messages):
    return "{}: {}".format(value.get(segment_fields.ID, "unknown id"), "; ".join(messages))


def format_bindings_error(value, messages):
    return "{}: {}".format(
        truncate(value if isinstance(value, (str, unicode)) else value[bindings.EXT_ID]),
        ";".join(messages)
    )


def truncate(string):
    max_length = 40
    if len(string) <= max_length:
        return string
    return string[:max_length] + " [truncated]"
