import datetime

from dateutil import relativedelta
import pytz

from crypta.buchhalter.services.main.lib.common import report_mail_sender
from crypta.dmp.yandex.bin.common.python import config_fields
from crypta.lib.native.yt.processed_tables_tracker.proto.tracked_source_pb2 import TTrackedSource
from crypta.lib.python import time_utils
from crypta.lib.python.smtp.text_mail_sender import TextMailSender
from crypta.lib.python.yt import path_utils
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


class DailySettings(object):
    date_format = "%Y-%m-%d"
    report_name_format = "%Y-%m-%d.xlsx"
    subject_template = "FTP user '{login}'. Daily usage report {date}"


class MonthlySettings(object):
    date_format = "%Y-%m"
    report_name_format = "%Y-%m.xlsx"
    subject_template = "FTP user '{login}'. Monthly usage report {date}"


def process(config, logger):
    yt_client = yt_helpers.get_yt_client(config[config_fields.YT_PROXY], config[config_fields.YT_POOL])
    mail_sender = TextMailSender(
        host=config[config_fields.SMTP_HOST],
        port=config[config_fields.SMTP_PORT],
        default_from_addr=config[config_fields.SMTP_EMAIL_FROM],
        default_to_addrs=config[config_fields.STATISTICS_EMAILS],
        default_cc=config[config_fields.REPORT_EMAILS_CC],
        default_bcc=config[config_fields.REPORT_EMAILS_BCC],
        dry_run=not config[config_fields.SEND_REPORT_EMAILS],
    )

    daily_reports_tracker = ProcessedTablesTracker(TTrackedSource(
        SourceDir=config[config_fields.DAILY_REPORTS_DIR],
        TrackTable=config[config_fields.SENT_DAILY_REPORTS_TRACK_TABLE],
    ))
    monthly_reports_tracker = ProcessedTablesTracker(TTrackedSource(
        SourceDir=config[config_fields.MONTHLY_REPORTS_DIR],
        TrackTable=config[config_fields.SENT_MONTHLY_REPORTS_TRACK_TABLE],
    ))

    dmp_login = config[config_fields.DMP_LOGIN]
    oldest_daily_report_date_to_send = time_utils.get_current_moscow_datetime() - datetime.timedelta(days=config[config_fields.OLDEST_DAILY_REPORT_TO_SEND_DAYS])
    oldest_monthly_report_date_to_send = time_utils.get_current_moscow_datetime() - relativedelta.relativedelta(months=config[config_fields.OLDEST_MONTHLY_REPORT_TO_SEND_MONTHS] + 1)

    logger.info("Process daily reports...")
    process_reports(
        tracker=daily_reports_tracker,
        yt_client=yt_client,
        mail_sender=mail_sender,
        dmp_login=dmp_login,
        oldest_report_date_to_send=oldest_daily_report_date_to_send,
        settings=DailySettings,
        logger=logger,
    )

    logger.info("Process monthly reports...")
    process_reports(
        tracker=monthly_reports_tracker,
        yt_client=yt_client,
        mail_sender=mail_sender,
        dmp_login=dmp_login,
        oldest_report_date_to_send=oldest_monthly_report_date_to_send,
        settings=MonthlySettings,
        logger=logger,
    )


def process_reports(tracker, yt_client, mail_sender, dmp_login, oldest_report_date_to_send, settings, logger):
    unprocessed_reports = tracker.get_unprocessed_tables(yt_client)

    if not unprocessed_reports:
        logger.info("No reports to process")
        return

    for unprocessed_report in unprocessed_reports:
        logger.info("Process %s", unprocessed_report)
        with yt_client.Transaction():
            filename = path_utils.get_basename(unprocessed_report)
            report_date = datetime.datetime.strptime(filename, settings.report_name_format).replace(tzinfo=pytz.timezone("Europe/Moscow"))

            if report_date < oldest_report_date_to_send:
                logger.info("Too old report. Skip sending")
            else:
                logger.info("Download report and send")
                report_mail_sender.send(mail_sender, yt_client, unprocessed_report, report_date, dmp_login, settings)

            tracker.add_processed_tables(yt_client, [unprocessed_report])
