import collections
from email.mime.text import MIMEText
import json

from crypta.dmp.common.data.python import segment_fields
from crypta.dmp.yandex.bin.common.python import (
    segment_action,
    statistics_fields
)
from crypta.lib.python import templater


SUBJECT_TEMPLATE = "FTP user '{dmp_login}'. Changes of meta in '{archive_name}'"
MESSAGE_TEMPLATE = """FTP user: {{dmp_login}}
Archive: {{archive_name}}
Some changes of segments meta info have been detected:
{% for change in changes %}
    * {{change}}
{% endfor %}

These changes will be applied soon.
"""

ACTION_TO_PRETTY = {
    segment_action.ADDED: "added",
    segment_action.DISABLED: "disabled",
    segment_action.REENABLED: "reenabled",
    segment_action.UPDATED: "updated",
    segment_action.UPDATED_AND_REENABLED: "updated and reenabled"
}


def generate(table_name, statistics, dmp_login):
    archive_ts, _ = table_name.split("_")
    archive_name = "segments-{}.tar.gz".format(archive_ts)

    aggregated_statistics = aggregate(statistics)

    subject = get_subject(dmp_login, archive_name)
    body = get_body(dmp_login, archive_name, aggregated_statistics)
    attachments = list(get_attachments(aggregated_statistics))

    return subject, body, attachments


def get_subject(dmp_login, archive_name):
    return SUBJECT_TEMPLATE.format(dmp_login=dmp_login, archive_name=archive_name)


def get_body(dmp_login, archive_name, aggregated_statistics):
    changes = []
    for action, data in aggregated_statistics.iteritems():
        changes.append("{} {} segment{}".format(len(data), ACTION_TO_PRETTY[action], "" if len(data) == 1 else "s"))

    return templater.render_template(MESSAGE_TEMPLATE, dict(
        dmp_login=dmp_login,
        archive_name=archive_name,
        changes=changes
    ))


def get_attachments(aggregated_statistics):
    for action, data in aggregated_statistics.iteritems():
        attachment = MIMEText(None, "plain")
        attachment.replace_header("content-transfer-encoding", "quoted-printable")
        attachment.add_header("Content-Disposition", "attachment", filename="{}.txt".format(action))
        attachment.set_payload(json.dumps(data, indent=4, ensure_ascii=False), "utf-8")
        yield attachment


def aggregate(statistics):
    aggregated_statistics = collections.defaultdict(list)
    for row in statistics:
        action = row[statistics_fields.ACTION]
        segment = row[statistics_fields.SEGMENT]
        diff = row[statistics_fields.DIFF]

        del segment[segment_fields.TIMESTAMP]
        del segment[segment_fields.STATUS]

        record = segment if not diff else {"changes": format_diff(diff), "new_state": segment}

        aggregated_statistics[action].append(record)

    return aggregated_statistics


def format_diff(diff):
    result = []

    for field, field_diff in diff.iteritems():
        result.append("{field}: {from_} -> {to}".format(
            field=field,
            from_=json.dumps(field_diff["from"], ensure_ascii=False),
            to=json.dumps(field_diff["to"], ensure_ascii=False)
        ))

    return result
