#include "update_bindings_reducer.h"

#include <crypta/dmp/common/serializers/bindings_serializer.h>

#include <util/generic/hash_set.h>

using namespace NCrypta::NDmp;

TUpdateBindingsReducer::TUpdateBindingsReducer(const TInputIndexes& inputIndexes, time_t bindingsTtl, const THashSet<ui64>& disabledSegments, const THashSet<ui64>& deletedSegments)
    : InputIndexes(inputIndexes)
    , BindingsTtl(bindingsTtl)
    , DisabledSegments(disabledSegments)
    , DeletedSegments(deletedSegments)
{
}

void TUpdateBindingsReducer::Do(TReader* reader, TWriter* writer) {
    TMaybe<TBindings> fresh;
    TMaybe<TBindings> state;
    Read(reader, fresh, state);
    if (fresh.Defined() && !BindingsTtl.IsExpired(fresh->Timestamp) && (state.Empty() || state->Timestamp < fresh->Timestamp)) {
        if (state.Defined()) {
            RestoreDisabled(fresh.GetRef(), state.GetRef());
        }
        WriteBindings(writer, fresh.GetRef());
    } else if (state.Defined() && !BindingsTtl.IsExpired(state->Timestamp)) {
        CleanupDeleted(state.GetRef());
        WriteBindings(writer, state.GetRef());
    }
}

void TUpdateBindingsReducer::Read(TReader* reader, TMaybe<TBindings>& fresh, TMaybe<TBindings>& state) const {
    for (; reader->IsValid(); reader->Next()) {
        const auto index = reader->GetTableIndex();
        if (index == InputIndexes[EInputTables::FreshBindings]) {
            fresh = NExtIdBindingsSerializer::Deserialize(reader->GetRow());
        } else if (index == InputIndexes[EInputTables::StateBindings]) {
            state = NExtIdBindingsSerializer::Deserialize(reader->GetRow());
        } else {
            ythrow yexception() << "Table index " << index << " not found in input indexes";
        }
    }
}

void TUpdateBindingsReducer::RestoreDisabled(TBindings& fresh, const TBindings& state) const {
    TSet<ui64> removedSegments;
    SetDifference(state.Segments.begin(), state.Segments.end(), fresh.Segments.begin(), fresh.Segments.end(),
                  std::inserter(removedSegments, removedSegments.begin()));
    for (const auto& removedSegment : removedSegments) {
        if (DisabledSegments.contains(removedSegment)) {
            fresh.Segments.insert(removedSegment);
        }
    }
}

void TUpdateBindingsReducer::CleanupDeleted(TBindings& state) const {
    for (auto it = state.Segments.begin(); it != state.Segments.end();) {
        if (DeletedSegments.contains(*it)) {
            it = state.Segments.erase(it);
        } else {
            ++it;
        }
    }
}

void TUpdateBindingsReducer::WriteBindings(TWriter* writer, const TBindings& bindings) const {
    writer->AddRow(NExtIdBindingsSerializer::Serialize(bindings));
}

REGISTER_REDUCER(TUpdateBindingsReducer);
