#include "update_meta_reducer.h"

#include <crypta/dmp/common/data/segment_fields.h>
#include <crypta/dmp/common/serializers/segment_serializer.h>
#include <crypta/dmp/yandex/bin/common/helpers.h>
#include <crypta/dmp/yandex/bin/common/segment_node_differ.h>
#include <crypta/dmp/yandex/bin/common/statistics_fields.h>

#include <util/string/builder.h>

using namespace NCrypta::NDmp;

TUpdateMetaReducer::TUpdateMetaReducer(const TInputIndexes& inputIndexes, const TOutputIndexes& outputIndexes, const TMaybe<ui64>& timestamp, time_t inactiveSegmentsTtl)
    : InputIndexes(inputIndexes)
    , OutputIndexes(outputIndexes)
    , Timestamp(timestamp)
    , InactiveSegmentsTtl(inactiveSegmentsTtl)
{
}

void TUpdateMetaReducer::Do(TReader* reader, TWriter* writer) {
    TMaybe<TSegment> fresh;
    TMaybe<TSegment> state;
    Read(reader, fresh, state);
    if (NeedToDisable(fresh, state)) {
        Disable(writer, state.GetRef());
    } else if (NeedToDelete(fresh, state)) {
        Delete(writer, state.GetRef());
    } else if (NeedToAdd(fresh, state)) {
        Add(writer, fresh.GetRef());
    } else if (NeedToUpdate(fresh, state)) {
        Update(writer, fresh.GetRef(), state.GetRef());
    } else {
        WriteMeta(writer, fresh.GetOrElse(state.GetRef()));
    }
}

void TUpdateMetaReducer::Read(TReader* reader, TMaybe<TSegment>& fresh, TMaybe<TSegment>& state) const {
    for (; reader->IsValid(); reader->Next()) {
        const auto index = reader->GetTableIndex();
        if (index == InputIndexes[EInputTables::FreshMeta]) {
            fresh = NSegmentSerializer::Deserialize(reader->GetRow());
        } else if (index == InputIndexes[EInputTables::StateMeta]) {
            state = NSegmentSerializer::Deserialize(reader->GetRow());
        } else {
            ythrow yexception() << "Table index " << index << " not found in input indexes";
        }
    }
}

bool TUpdateMetaReducer::NeedToDisable(const TMaybe<TSegment>& fresh, const TMaybe<TSegment>& state) const {
    return InputIndexes[EInputTables::FreshMeta].Defined() && !fresh.Defined() && state.Defined() && state->IsEnabled();
}

bool TUpdateMetaReducer::NeedToDelete(const TMaybe<TSegment>& fresh, const TMaybe<TSegment>& state) const {
    return !fresh.Defined() && state.Defined() && state->IsDisabled() && InactiveSegmentsTtl.IsExpired(state->Timestamp);
}

bool TUpdateMetaReducer::NeedToAdd(const TMaybe<TSegment>& fresh, const TMaybe<TSegment>& state) const {
    return fresh.Defined() && !state.Defined();
}

bool TUpdateMetaReducer::NeedToUpdate(const TMaybe<TSegment>& fresh, const TMaybe<TSegment>& state) const {
    return fresh.Defined() && state.Defined();
}

void TUpdateMetaReducer::Disable(TWriter* writer, TSegment& state) const {
    state.SetDisabled();
    state.Timestamp = Timestamp.GetRef();
    WriteStatistics(writer, state, ESegmentAction::Disabled);
    WriteMeta(writer, state);
}

void TUpdateMetaReducer::Delete(TWriter* writer, TSegment& state) const {
    state.SetDeleted();
    WriteMeta(writer, state);
}

void TUpdateMetaReducer::Add(TWriter* writer, const TSegment& fresh) const {
    WriteStatistics(writer, fresh, ESegmentAction::Added);
    WriteMeta(writer, fresh);
}

void TUpdateMetaReducer::Update(TWriter* writer, const TSegment& fresh, const TSegment& state) const {
    if (state.Timestamp > fresh.Timestamp) {
        WriteMeta(writer, state);
        return;
    }

    const NYT::TNode diff = DiffSegments(state, fresh);
    if (diff.HasKey(NSegmentFields::TARIFF)) {
         const TString msg = TStringBuilder() << "Tariff change from " << state.Tariff << " to " << fresh.Tariff << " is forbidden";
         const NYT::TNode error = CreateError("meta", NSegmentSerializer::Serialize(fresh), msg);
         writer->AddRow(error, OutputIndexes[EOutputTables::Errors].GetRef());
    }

    if (!diff.Empty()) {
        ESegmentAction action;
        if (diff.Size() == 1 && diff.HasKey(NSegmentFields::STATUS)) {
            action = ESegmentAction::Reenabled;
        } else if (diff.HasKey(NSegmentFields::STATUS)) {
            action = ESegmentAction::UpdatedAndReenabled;
        } else {
            action = ESegmentAction::Updated;
        }

        WriteStatistics(writer, fresh, action, diff);
    }

    WriteMeta(writer, fresh);
}

void TUpdateMetaReducer::WriteMeta(TWriter* writer, const TSegment& segment) const {
    writer->AddRow(NSegmentSerializer::Serialize(segment), OutputIndexes[EOutputTables::Meta].GetRef());
}

void TUpdateMetaReducer::WriteStatistics(TWriter* writer, const TSegment& segment, ESegmentAction action, const NYT::TNode& diff) const {
    using namespace NStatisticsFields;
    const NYT::TNode row = NYT::TNode()
        (SEGMENT, NSegmentSerializer::Serialize(segment))
        (ACTION, ToString(action))
        (DIFF, diff);
    writer->AddRow(row, OutputIndexes[EOutputTables::Statistics].GetRef());
}

REGISTER_REDUCER(TUpdateMetaReducer);
