#include "update_bindings_reducer.h"
#include "update_meta_reducer.h"
#include "update_segments_job.h"

#include <crypta/dmp/common/data/bindings_id_fields.h>
#include <crypta/dmp/common/data/bindings_schema.h>
#include <crypta/dmp/common/data/meta_schema.h>
#include <crypta/dmp/common/data/segment_fields.h>
#include <crypta/dmp/common/helpers/meta.h>
#include <crypta/dmp/yandex/bin/common/errors_schema.h>
#include <crypta/dmp/yandex/bin/common/fresh_segments.h>
#include <crypta/dmp/yandex/bin/common/helpers.h>
#include <crypta/dmp/yandex/bin/common/statistics_schema.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_suffix_generator.h>
#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/util/ypath_join.h>
#include <util/generic/hash_set.h>
#include <util/string/join.h>

using namespace NCrypta::NDmp;

TUpdateSegmentsJob::TUpdateSegmentsJob(const TConfig& config, NLog::TLogPtr log)
    : Config(config)
    , Log(log)
{
    Y_ENSURE(log != nullptr);
}

int TUpdateSegmentsJob::Do() const {
    Log->info("================ Start ================");
    NYT::TConfig::Get()->Pool = Config.YtPool;
    auto client = NYT::CreateClient(Config.YtProxy);

    Log->info("Look for fresh segments in {}", Config.InputDir);
    const auto listOfFreshSegments = TFreshSegments::List(client, Config.InputDir, TTimedSuffixGenerator::RemoveSuffix<ui64>);
    Log->info("List of fresh segments [\n{}\n]", JoinSeq(",\n", listOfFreshSegments));

    if (listOfFreshSegments.empty()) {
        Log->info("No fresh segments. Just update state");
        auto tx = client->StartTransaction();
        UpdateSegments(tx, Nothing());
        tx->Commit();
    }

    for (const auto& freshSegments : listOfFreshSegments) {
        Log->info("Update state with {}", ToString(freshSegments));
        auto tx = client->StartTransaction();
        UpdateSegments(tx, freshSegments);
        if (Config.ShouldDropInput) {
            Log->info("Remove {} with content", freshSegments.Dir);
            freshSegments.Remove(tx);
        }
        tx->Commit();
    }

    Log->info("================ Finish ================");
    return 0;
}

void TUpdateSegmentsJob::UpdateSegments(NYT::ITransactionPtr tx, const TMaybe<TFreshSegments>& freshSegments) const {
    const TMaybe<NYT::TYPath> stateMeta = tx->Exists(Config.MetaState) ? MakeMaybe(Config.MetaState) : Nothing();
    const TMaybe<NYT::TYPath> stateBindings = tx->Exists(Config.BindingsState) ? MakeMaybe(Config.BindingsState) : Nothing();
    Y_ENSURE(stateMeta.Defined() || !stateBindings.Defined());

    if (!stateMeta.Defined() && !freshSegments.Defined()) {
        Log->info("Nothing to do");
        return;
    }

    TMaybe<ui64> timestamp = freshSegments.Defined() ? MakeMaybe(freshSegments->Timestamp) : Nothing();
    TMaybe<NYT::TYPath> freshMeta = freshSegments.Defined() ? MakeMaybe(freshSegments->Meta) : Nothing();
    TMaybe<NYT::TYPath> freshBindings = freshSegments.Defined() ? freshSegments->Bindings : Nothing();

    NYT::TTempTable tmpMeta(tx);
    NYT::TTempTable tmpStatistics(tx);
    bool noErrors = UpdateMeta(tx, timestamp, freshMeta, stateMeta, tmpMeta.Name(), tmpStatistics.Name());
    if (!noErrors) {
        return;
    }

    TMaybe<NYT::TTempTable> tmpBindings;
    if (freshBindings.Defined() || stateBindings.Defined()) {
        tmpBindings = NYT::TTempTable(tx);
        const THashSet<ui64> disabledSegments = GetSegmentsIds(tx, tmpMeta.Name(), [](const TSegment& segment) {
            return segment.IsDisabled();
        });
        const THashSet<ui64> deletedSegments = GetSegmentsIds(tx, tmpMeta.Name(), [](const TSegment& segment) {
            return segment.IsDeleted();
        });
        UpdateBindings(tx, freshBindings, stateBindings, tmpBindings->Name(), disabledSegments, deletedSegments);
    }

    Log->info("Put meta to {}", Config.MetaState);
    tx->Copy(tmpMeta.Name(), Config.MetaState, NYT::TCopyOptions().Force(true).Recursive(true));
    if (tmpBindings.Defined()) {
        Log->info("Put bindings to {}", Config.BindingsState);
        tx->Copy(tmpBindings->Name(), Config.BindingsState, NYT::TCopyOptions().Force(true).Recursive(true));
    }
    if (freshSegments.Defined()) {
        const TString statisticsTableName = TTimedSuffixGenerator().AppendSuffix(freshSegments->Timestamp);
        const auto statisticsTablePath = NYT::JoinYPaths(Config.StatisticsOutputDir, statisticsTableName);
        Log->info("Put statistics to {}", statisticsTablePath);
        tx->Copy(tmpStatistics.Name(), statisticsTablePath, NYT::TCopyOptions().Recursive(true));
    }
}


bool TUpdateSegmentsJob::UpdateMeta(
        NYT::ITransactionPtr tx,
        TMaybe<ui64>& timestamp,
        const TMaybe<NYT::TYPath>& fresh,
        const TMaybe<NYT::TYPath>& state,
        const NYT::TYPath& output,
        const NYT::TYPath& statistics) const
{
    Log->info("Update state meta {} with fresh meta {}", ToString(state), ToString(fresh));
    NYT::TTempTable errors(tx);

    TUpdateMetaReducer::TInputIndexes::TBuilder inputBuilder;
    if (fresh.Defined()) {
        inputBuilder.Add(fresh.GetRef(), TUpdateMetaReducer::EInputTables::FreshMeta);
    }
    if (state.Defined()) {
        inputBuilder.Add(state.GetRef(), TUpdateMetaReducer::EInputTables::StateMeta);
    }

    TUpdateMetaReducer::TOutputIndexes::TBuilder outputBuilder;
    outputBuilder.Add(NYT::TRichYPath(output).Schema(GetMetaSchema()).OptimizeFor(NYT::OF_SCAN_ATTR), TUpdateMetaReducer::EOutputTables::Meta);
    outputBuilder.Add(NYT::TRichYPath(statistics).Schema(GetStatisticsSchema()).OptimizeFor(NYT::OF_SCAN_ATTR), TUpdateMetaReducer::EOutputTables::Statistics);
    outputBuilder.Add(NYT::TRichYPath(errors.Name()).Schema(GetErrorsSchema()).OptimizeFor(NYT::OF_SCAN_ATTR), TUpdateMetaReducer::EOutputTables::Errors);

    auto spec = NYT::TReduceOperationSpec().ReduceBy(NSegmentFields::ID);
    AddInputs<NYT::TNode>(spec, inputBuilder.GetTables());
    AddOutputs<NYT::TNode>(spec, outputBuilder.GetTables());

    tx->Reduce(spec, new TUpdateMetaReducer(inputBuilder.GetIndexes(), outputBuilder.GetIndexes(), timestamp, Config.InactiveSegmentsTtl));

    if (!EmptyTable(tx, errors.Name())) {
        Y_ENSURE(fresh.Defined());
        MoveToQuarantine(tx, errors.Name(), Config.QuarantineDir, timestamp.GetRef());
        Log->warn("Meta is not updated. See quarantine");
        return false;
    }

    Log->info("Meta is updated");
    return true;
}

void TUpdateSegmentsJob::UpdateBindings(
    NYT::ITransactionPtr tx,
    const TMaybe<NYT::TYPath>& fresh,
    const TMaybe<NYT::TYPath>& state,
    const NYT::TYPath& output,
    const THashSet<ui64>& disabledSegments,
    const THashSet<ui64>& deletedSegments) const
{
    Log->info("Update state bindings {} with fresh bindings {}", ToString(state), ToString(fresh));

    TUpdateBindingsReducer::TInputIndexes::TBuilder inputBuilder;
    if (fresh.Defined()) {
        inputBuilder.Add(fresh.GetRef(), TUpdateBindingsReducer::EInputTables::FreshBindings);
    }
    if (state.Defined()) {
        inputBuilder.Add(state.GetRef(), TUpdateBindingsReducer::EInputTables::StateBindings);
    }

    const auto richOutput = NYT::TRichYPath(output).Schema(GetExtIdBindingsSchema()).OptimizeFor(NYT::OF_SCAN_ATTR);

    auto spec = NYT::TReduceOperationSpec()
        .ReduceBy(NBindingsIdFields::EXT_ID)
        .AddOutput<NYT::TNode>(richOutput);
    AddInputs<NYT::TNode>(spec, inputBuilder.GetTables());

    tx->Reduce(spec, new TUpdateBindingsReducer(inputBuilder.GetIndexes(), Config.BindingsTtl, disabledSegments, deletedSegments));
    Log->info("Bindings is updated");
}
