import datetime
import os

from dateutil import relativedelta
import pytz

from crypta.dmp.yandex.bin.common.python import config_fields
from crypta.lib.native.yt.processed_tables_tracker.proto.tracked_source_pb2 import TTrackedSource
from crypta.lib.python import time_utils
from crypta.lib.python.ftp.client.ftp_client import FtpClient
from crypta.lib.python.yt import (
    path_utils,
    yt_helpers,
)
from crypta.lib.python.yt.processed_tables_tracker import ProcessedTablesTracker


REPORTS_DIR = "reports"


class DailySettings(object):
    report_name_format = "%Y-%m-%d.xlsx"
    output_dir = "daily"


class MonthlySettings(object):
    report_name_format = "%Y-%m.xlsx"
    output_dir = "monthly"


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config[config_fields.YT_PROXY], config[config_fields.YT_POOL])
    ftp_client = FtpClient(
        config[config_fields.FTP_HOST],
        config[config_fields.FTP_PORT],
        config[config_fields.FTP_USER],
        os.environ["CRYPTA_DMP_FTP_PASSWORD"],
        config[config_fields.FTP_DIR],
    )
    daily_reports_tracker = ProcessedTablesTracker(TTrackedSource(
        SourceDir=config[config_fields.DAILY_REPORTS_DIR],
        TrackTable=config[config_fields.UPLOADED_DAILY_REPORTS_TRACK_TABLE],
    ))
    monthly_reports_tracker = ProcessedTablesTracker(TTrackedSource(
        SourceDir=config[config_fields.MONTHLY_REPORTS_DIR],
        TrackTable=config[config_fields.UPLOADED_MONTHLY_REPORTS_TRACK_TABLE],
    ))
    oldest_daily_report_date_to_upload = time_utils.get_current_moscow_datetime() - datetime.timedelta(days=config[config_fields.OLDEST_DAILY_REPORT_TO_UPLOAD_DAYS])
    oldest_monthly_report_date_to_upload = time_utils.get_current_moscow_datetime() - relativedelta.relativedelta(months=config[config_fields.OLDEST_MONTHLY_REPORT_TO_UPLOAD_MONTHS] + 1)
    dry_run = not config[config_fields.UPLOAD_REPORTS_TO_FTP]

    logger.info("Process daily reports...")
    process_reports(
        tracker=daily_reports_tracker,
        yt_client=yt_client,
        ftp_client=ftp_client,
        oldest_report_date_to_send=oldest_daily_report_date_to_upload,
        dry_run=dry_run,
        settings=DailySettings,
        logger=logger,
    )

    logger.info("Process monthly reports...")
    process_reports(
        tracker=monthly_reports_tracker,
        yt_client=yt_client,
        ftp_client=ftp_client,
        oldest_report_date_to_send=oldest_monthly_report_date_to_upload,
        dry_run=dry_run,
        settings=MonthlySettings,
        logger=logger,
    )


def process_reports(tracker, yt_client, ftp_client, oldest_report_date_to_send, dry_run, settings, logger):
    unprocessed_reports = tracker.get_unprocessed_tables(yt_client)

    if not unprocessed_reports:
        logger.info("No reports to process")
        return

    for unprocessed_report in unprocessed_reports:
        logger.info("Process %s", unprocessed_report)
        with yt_client.Transaction():
            filename = path_utils.get_basename(unprocessed_report)
            report_date = datetime.datetime.strptime(filename, settings.report_name_format).replace(tzinfo=pytz.timezone("Europe/Moscow"))

            if report_date < oldest_report_date_to_send:
                logger.info("Too old report. Skip uploading")
            else:
                logger.info("Upload report to ftp")

                with open(filename, "wb") as f:
                    f.write(yt_client.read_file(unprocessed_report).read())

                destination_dir = os.path.join(REPORTS_DIR, settings.output_dir)
                destination_path = os.path.join(destination_dir, filename)

                if dry_run:
                    logger.info("Skip actual uploading")
                else:
                    ftp_client.mkd(REPORTS_DIR, ignore_error_perm=True)
                    ftp_client.mkd(destination_dir, ignore_error_perm=True)
                    ftp_client.upload(filename, destination_path)

                os.remove(filename)

            tracker.add_processed_tables(yt_client, [unprocessed_report])
