# -*- coding: utf-8 -*-
import argparse
import datetime
import logging

import yt.wrapper as yt

from crypta.dmp.common.upload_to_audience import (
    flatten_segments_query,
    upload,
)
from crypta.dmp.yandex.bin.common.python import config_fields
from crypta.lib.python import yaml_config
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.yql import executer
from crypta.lib.python.yt import (
    path_utils,
    yt_helpers,
)


logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser(description="Uploads free DMP segments with ACL to Yandex.Audience")
    parser.add_argument("--config", help="Config file path", required=True, type=yaml_config.load)
    return parser.parse_args()


QUERY_TEMPLATE = """
$free_tariffs = (
    SELECT
        tariff_id as tariff
    FROM `{tariff_prices_table}`
    WHERE price_rub == 0 AND dmp_id == {dmp_id}
);

$private_segments_meta = (
    SELECT
        meta.*
    FROM `{meta_table}` as meta
    INNER JOIN $free_tariffs as free_tariffs
    ON meta.tariff == free_tariffs.tariff
    WHERE meta.acl IS NOT NULL AND ListLength(Yson::ConvertToStringList(meta.acl)) != 0
);

$flattened_segments = (
    {flatten_segments_query}
);

INSERT INTO `{dst_table}` WITH TRUNCATE
SELECT
    flattened_segments.id as id,
    flattened_segments.yandexuid as yandexuid
FROM $flattened_segments as flattened_segments
INNER JOIN $private_segments_meta as private_segments_meta
ON flattened_segments.id == private_segments_meta.id
ORDER BY id;

INSERT INTO `{private_segments_meta}` WITH TRUNCATE
SELECT
    *
FROM $private_segments_meta
ORDER BY id;
"""


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    logger.info("args: %s", args)

    yt_proxy = args.config[config_fields.YT_PROXY]
    yt_pool = args.config[config_fields.YT_POOL]
    yt_tmp_dir = args.config[config_fields.YT_TMP_DIR]

    yt.config.set_proxy(yt_proxy)
    yt.config["pool"] = yt_pool

    meta_table_path = args.config[config_fields.META_TABLE]

    with yt.Transaction() as tx, yt.TempTable(yt_tmp_dir) as flattened_segments_table, yt.TempTable(yt_tmp_dir) as private_segments_meta:
        query = QUERY_TEMPLATE.format(dmp_id=args.config[config_fields.DMP_ID],
                                      tariff_prices_table=args.config[config_fields.TARIFF_PRICES_TABLE],
                                      meta_table=meta_table_path,
                                      dst_table=flattened_segments_table,
                                      private_segments_meta=private_segments_meta,
                                      flatten_segments_query=flatten_segments_query.QUERY_TEMPLATE.format(bindings_table=args.config[config_fields.YANDEXUID_BINDINGS_TABLE]))

        logger.info("Running YQL query:\n%s", query)
        executer.get_executer(yt_proxy, yt_pool, yt_tmp_dir)(query, transaction=tx.transaction_id, syntax_version=1)

        errors_table_path = path_utils.get_ts_table_path(args.config[config_fields.UPLOAD_TO_AUDIENCE_ERRORS_DIR])
        upload.upload_flattened_bindings(flattened_segments_table,
                                         private_segments_meta,
                                         args.config[config_fields.AUDIENCE_LOGIN],
                                         errors_table_path,
                                         args.config[config_fields.UPLOAD_TO_AUDIENCE_MEMORY_LIMIT],
                                         args.config[config_fields.AUDIENCE_SRC_TVM_ID],
                                         args.config[config_fields.AUDIENCE_API_DST_TVM_ID],
                                         max_concurrent_jobs=args.config[config_fields.UPLOAD_TO_AUDIENCE_MAX_CONCURRENT_JOBS],
                                         min_segment_size=args.config[config_fields.AUDIENCE_MIN_SEGMENT_SIZE],
                                         max_segment_size=args.config[config_fields.AUDIENCE_MAX_SEGMENT_SIZE],
                                         audience_url=args.config[config_fields.AUDIENCE_URL],
                                         audience_port=args.config[config_fields.AUDIENCE_PORT])

    yt_helpers.set_ttl(errors_table_path, datetime.timedelta(days=args.config[config_fields.UPLOAD_TO_AUDIENCE_ERRORS_TTL_DAYS]))
    logger.info("Completed successfully")
