import datetime
import os

import yaml

from crypta.dmp.common import index_status
import crypta.dmp.common.check_upload_to_audience_errors.lib.check as upload_to_audience_check
from crypta.dmp.yandex.bin.common.python import config_fields
from crypta.dmp.yandex.spine import common
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.sandbox import (
    configs,
    sandbox_scheduler,
)
from crypta.lib.python.spine.solomon import (
    solomon_alert_utils,
    solomon_check_generator,
)
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_latency_metric import YtLatencyMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from crypta.lib.python import templater
from sandbox.projects.crypta import merge_to_bigb_collector
from sandbox.projects.crypta.dmp import (
    check_upload_to_audience_errors,
)
from sandbox.projects.crypta.dmp.yandex import (
    calc_sizes,
    ftp_to_yt,
    make_mac_hash_yuid,
    process_segments,
    publish_index,
    send_metrics,
    send_metrics_mail,
    send_quarantine_mail,
    send_statistics_mail,
    send_report_mail,
    upload_report_to_ftp,
    upload_to_audience,
)


_WITH_BIG_ARCHIVES = ["dmp-amberdata"]


def render_path(prefix, login, suffix):
    return os.path.join(prefix, suffix).format(dmp=login)


def add_disk_space_metric(yt_registry, path, threshold):
    metric = YtSizeMetric(yt_registry, path)
    if threshold is not None:
        metric.add_disk_space_alert(
            predicate=alert_pb2.GT,
            threshold=threshold,
        )


def get_index(env):
    index = yaml.safe_load(templater.render_resource("/dmp_index", vars={"environment": env}, strict=True))
    return [dmp for dmp in index if dmp["status"] == index_status.ACTIVE and not dmp.get("internal")]


def get_registry():
    juggler_host = "crypta-dmp-yandex"
    base_juggler = juggler_check_generator.CryptaYtCheckGenerator(
        tags=["crypta-dmp-yandex"],
        host=juggler_host,
        child_group_type=consts.GroupType.host,
        child_group="{}-raw".format(juggler_host),
    )

    add_general_yt_metrics(base_juggler)
    add_general_sandbox_tasks(base_juggler)

    for env in (environment.TESTING, environment.STABLE):
        index = get_index(env)
        for item in index:
            dmp_login = item["dmp_login"]
            juggler = base_juggler.clone(tags=["crypta-{}".format(dmp_login)])

            add_per_login_sandbox_tasks(juggler, dmp_login, env)

            if env == environment.STABLE:
                add_per_login_yt_metrics(juggler, dmp_login)
                add_per_login_solomon_checks(juggler, dmp_login)
                add_per_login_upload_to_audience_check_errors(juggler, item)

    return base_juggler


def add_general_yt_metrics(base_juggler):
    general_yt_sizes_dashboard_paths = [
        ("common/dmp_mac_hash_yuid", None),
    ]

    base_yt_registry = yt_config_registry.DeprecatedCryptaYtConfigRegistry(base_juggler)

    for path, threshold in general_yt_sizes_dashboard_paths:
        add_disk_space_metric(base_yt_registry, path, threshold)


def add_general_sandbox_tasks(base_juggler):
    sandbox = sandbox_scheduler.create_default_generator(base_juggler, ["DMP"])
    juggler_service_suffix = "dmp-segments"

    for task_config in (
        configs.TaskConfig(
            publish_index.CryptaDmpYandexPublishIndexTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=16),
                kill_timeout=datetime.timedelta(minutes=20),
                schedule_interval=datetime.timedelta(hours=1),
            ),
            testing_from_stable=True,
        ),
        configs.TaskConfig(
            check_upload_to_audience_errors.CryptaDmpCheckUploadToAudienceErrorsTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=2),
                kill_timeout=datetime.timedelta(minutes=30),
                schedule_interval=datetime.timedelta(minutes=30),
            ),
        ),
        configs.TaskConfig(
            merge_to_bigb_collector.CryptaMergeToBigbCollectorTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=2),
                kill_timeout=datetime.timedelta(hours=1),
                schedule_interval=datetime.timedelta(minutes=15),
                extra_params={
                    "pool": "crypta_dmp",
                    "fresh_dir": "//home/crypta/production/dmp/bb_collector/fresh",
                    "output_dir": "//home/crypta/production/dmp/bb_collector/output",
                    "juggler_service_suffix": juggler_service_suffix
                },
                juggler_service=merge_to_bigb_collector.CryptaMergeToBigbCollectorTask.get_juggler_service_with_suffix(juggler_service_suffix),
            ),
        ),
        configs.TaskConfig(
            make_mac_hash_yuid.CryptaDmpYandexMakeMacHashYuidTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=16),
                kill_timeout=datetime.timedelta(hours=2),
                schedule_interval=datetime.timedelta(hours=2),
            ),
            testing_from_stable=True,
        ),
    ):
        for config in task_config:
            scheduler = sandbox.create_scheduler(**config.get_scheduler_config())
            if config.env == environment.STABLE:
                scheduler.check(**config.get_check_config()).add_yt_dependencies(YtProxy.Group.offline)


def add_per_login_sandbox_tasks(juggler, dmp_login, env):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["DMP", dmp_login.upper()])

    sandbox_tasks = [
        configs.TaskConfig(
            ftp_to_yt.CryptaDmpYandexFtpToYtTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=24),
                kill_timeout=datetime.timedelta(hours=get_ftp_to_yt_ttk_hours(dmp_login)),
                schedule_interval=datetime.timedelta(minutes=30),
            ),
            testing_from_stable=True,
        ),
        configs.TaskConfig(
            process_segments.CryptaDmpYandexProcessSegmentsTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=24),
                kill_timeout=datetime.timedelta(hours=10),
                schedule_interval=datetime.timedelta(hours=1),
            ),
            testing=configs.RunConfig(
                kill_timeout=datetime.timedelta(hours=10),
                schedule_interval=datetime.timedelta(hours=6),
            ),
        ),
        configs.TaskConfig(
            calc_sizes.CryptaDmpYandexCalcSizesTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=24),
                kill_timeout=datetime.timedelta(minutes=90),
                schedule_interval=datetime.timedelta(hours=1),
            ),
            testing=configs.RunConfig(
                kill_timeout=datetime.timedelta(hours=3),
                schedule_interval=datetime.timedelta(hours=6),
            ),
        ),
        configs.TaskConfig(
            send_metrics.CryptaDmpYandexSendMetricsTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=24),
                kill_timeout=datetime.timedelta(minutes=40),
                schedule_interval=datetime.timedelta(hours=1),
            ),
            testing=configs.RunConfig(
                kill_timeout=datetime.timedelta(hours=1),
                schedule_interval=datetime.timedelta(hours=4),
            ),
        ),
        configs.TaskConfig(
            send_quarantine_mail.CryptaDmpYandexSendQuarantineMailTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=24),
                kill_timeout=datetime.timedelta(minutes=20),
                schedule_interval=datetime.timedelta(minutes=15),
            ),
            testing_from_stable=True,
        ),
        configs.TaskConfig(
            send_statistics_mail.CryptaDmpYandexSendStatisticsMailTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=24),
                kill_timeout=datetime.timedelta(minutes=30),
                schedule_interval=datetime.timedelta(hours=1),
            ),
            testing_from_stable=True,
        ),
        configs.TaskConfig(
            send_metrics_mail.CryptaDmpYandexSendMetricsMailTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=25),
                kill_timeout=datetime.timedelta(minutes=90),
                schedule_daily_start_time="2019-02-28T09:00:00Z",
                retry_interval=datetime.timedelta(hours=1),
            ),
            testing_from_stable=True,
        ),
        configs.TaskConfig(
            send_report_mail.CryptaDmpYandexSendReportMailTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=25),
                kill_timeout=datetime.timedelta(minutes=30),
                schedule_interval=datetime.timedelta(hours=1),
            ),
            testing=configs.RunConfig(
                kill_timeout=datetime.timedelta(minutes=20),
                schedule_interval=datetime.timedelta(hours=4),
            ),
        ),
        configs.TaskConfig(
            upload_report_to_ftp.CryptaDmpYandexUploadReportToFtpTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=25),
                kill_timeout=datetime.timedelta(hours=1),
                schedule_interval=datetime.timedelta(hours=1),
            ),
            testing_from_stable=True,
        ),
        configs.TaskConfig(
            upload_to_audience.CryptaDmpYandexUploadToAudienceTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(hours=24),
                kill_timeout=datetime.timedelta(hours=12),
                schedule_interval=datetime.timedelta(hours=4),
            ),
        ),
    ]
    for task_config in sandbox_tasks:
        for config in task_config:
            if config.env == env:
                config.extra_params = {"login": dmp_login}
                scheduler = sandbox.create_scheduler(**config.get_scheduler_config())
                if config.env == environment.STABLE:
                    scheduler.check(config.crit_time, juggler_service=config.task_cls.get_juggler_service_for_login(dmp_login)).add_yt_dependencies(YtProxy.Group.offline)


def add_per_login_yt_metrics(juggler, dmp_login):
    prefix = "dmp/yandex/clients/{dmp}"

    yt_sizes_dashboard_paths = [
        ("ext_id_bindings", None),
        ("unmerged_yandexuid_bindings", None),
        ("yandexuid_bindings", None),
        ("meta", None),
        ("raw_segments", None),
        ("parsed_segments", None),
        ("statistics", None),
        ("quarantine", None),
        ("errors/upload_to_audience", DataSize(mb=1)),
    ]

    yt_latencies_dashboard_paths = [
        ("raw_segments", _get_raw_segments_latency(dmp_login)),
        ("parsed_segments", datetime.timedelta(hours=3)),
        ("quarantine", datetime.timedelta(hours=3)),
        ("statistics", datetime.timedelta(hours=3)),
    ]

    yt_registry = yt_config_registry.DeprecatedCryptaYtConfigRegistry(juggler)

    for path, threshold in yt_sizes_dashboard_paths:
        add_disk_space_metric(yt_registry, render_path(prefix, dmp_login, path), threshold)

    for path, threshold in yt_latencies_dashboard_paths:
        YtLatencyMetric(yt_registry, render_path(prefix, dmp_login, path)).add_latency_alert(
            threshold=threshold,
        )


def add_per_login_solomon_checks(juggler, dmp_login):
    solomon = solomon_check_generator.SolomonCheckGenerator(
        juggler,
        solomon_alert_utils.AlertCreator("crypta_dmp", {
            "project": "crypta_dmp",
            "service": "stats",
            "cluster": environment.PRODUCTION
        })
    )

    for sensor, description, flap_detector in [
        ("segments_size.deleted.ext_id", "Deleted segments size (ext_id): {{ pointValue }}", None),
        ("segments_size.deleted.yuid", "Deleted segments size (yandexuid): {{ pointValue }}", FlapDetectorParams(
            stable_time=datetime.timedelta(hours=8),
            crit_time=datetime.timedelta(hours=16),
        )),
    ]:
        check = solomon.get_sensor({"dmp": dmp_login, "sensor": sensor}).create_threshold_check(
            aggregation=alert_pb2.MAX,
            predicate=alert_pb2.GT,
            threshold=0,
            period=datetime.timedelta(hours=4),
            description=description,
            juggler_service=common.get_raw_juggler_service("{}__{}".format(common.get_short_dmp_name(dmp_login), sensor)),
        ).add_yt_dependencies([YtProxy.hahn])

        if flap_detector is not None:
            check.add_flap_detector(flap_detector)


def add_per_login_upload_to_audience_check_errors(juggler, item):
    config = yaml.safe_load(templater.render_resource("/dmp_config", vars=dict(environment=environment.STABLE, **item), strict=True))
    upload_to_audience_check.add(juggler, config[config_fields.UPLOAD_TO_AUDIENCE_ERRORS_DIR], common.get_short_dmp_name(item["dmp_login"]))


def get_ftp_to_yt_ttk_hours(dmp_login):
    return 6 if dmp_login in _WITH_BIG_ARCHIVES else 3


def _get_raw_segments_latency(dmp_login):
    hours = (7 if dmp_login in _WITH_BIG_ARCHIVES else 3)
    return datetime.timedelta(hours=hours)
