#include <crypta/ext_fp/common/unprocessed_queue/unprocessed_queue.h>
#include <crypta/ext_fp/delay_line/bin/cmd_args.pb.h>
#include <crypta/ext_fp/delay_line/lib/config/config.pb.h>
#include <crypta/ext_fp/delay_line/lib/parser.h>
#include <crypta/ext_fp/delay_line/lib/processor.h>
#include <crypta/lib/native/cmd_args/parse_pb_options.h>
#include <crypta/lib/native/log/utils/log_exception.h>
#include <crypta/lib/native/pqlib/consumer.h>
#include <crypta/lib/native/pqlib/cookie_queue.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/logger.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/producer.h>
#include <crypta/lib/native/pqlib/reader.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/graphite/graphite_stats_sink.h>
#include <crypta/lib/native/stats/log/log_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yaml/parse_yaml_file.h>
#include <crypta/lib/native/yaml/yaml2proto.h>

#include <yt/yt/core/concurrency/thread_pool.h>
#include <yt/yt/core/misc/shutdown.h>

#include <util/generic/vector.h>
#include <util/stream/output.h>

using namespace NCrypta;
using namespace NCrypta::NExtFp;
using namespace NCrypta::NExtFp::NDelayLine;
using namespace NCrypta::NLog;

namespace {
    TConfig ParseArgs(int argc, const char** argv) {
        const auto& options = ParsePbOptions<TCmdArgs>(argc, argv);
        auto yamlConfig = ParseYamlFile(options.GetConfigFile());

        return Yaml2Proto<TConfig>(yamlConfig);
    }
}

int main(int argc, const char** argv) {
    TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

    try {
        TShiftedClock::FreezeTimestampFromEnv();

        const auto& config = ParseArgs(argc, argv);

        RegisterLogs(config.GetLogs());

        auto log = NLog::GetLog("main");

        auto statsRegistry = Singleton<TStatsRegistry>();
        TStatsWriter statsWriter(*statsRegistry, TDuration::Minutes(1));
        if (config.HasGraphite()) {
            statsWriter.AddSink(MakeHolder<TGraphiteStatsSink>(config.GetGraphite()));
        }
        if (config.HasSolomon()) {
            statsWriter.AddSink(MakeHolder<TSolomonStatsSink>(config.GetSolomon()));
        }
        statsWriter.AddSink(MakeHolder<TLogStatsSink>("graphite"));
        statsWriter.Start();

        TStats::TSettings statsSettings;
        statsSettings.HistMax = 10;
        statsSettings.HistBinCount = 10;

        TStats pqlibStats("pqlib", statsSettings);
        TStats processorStats("processor", statsSettings);

        const auto& logbrokerConfig = config.GetLogbroker();
        auto pqLibLogger = NPQ::NLogger::Create(logbrokerConfig.GetPqlibLogName());

        auto pqlib = NPQ::NPQLib::Create(logbrokerConfig.GetPqlib());

        NPQ::TConsumer consumer(
                pqlib,
                pqLibLogger,
                logbrokerConfig.GetConsumer(),
                NPQ::NCredentialsProvider::Create(logbrokerConfig.GetCredentials(), pqLibLogger),
                pqlibStats
        );

        const auto& outputLogProducerConfig = logbrokerConfig.GetExtFpEventLogProducer();
        NPQ::TProducer outputLogProducer(
                pqlib,
                outputLogProducerConfig,
                NPQ::NCredentialsProvider::Create(logbrokerConfig.GetCredentials(), NPQ::NLogger::Create(outputLogProducerConfig.GetLogName())),
                statsSettings
        );

        TUnprocessedQueue<TVector<NCrypta::NExtFp::TFpEvent>> unprocessedQueue;

        TVector<THolder<TProcessor>> processors;
        for (ui32 i = 0; i < config.GetProcessorCount(); ++i) {
            processors.emplace_back(MakeHolder<TProcessor>(
                    unprocessedQueue,
                    config.GetProcessor(),
                    outputLogProducer,
                    processorStats));
        }


        NPQ::TCookieQueue cookiesToCommit;
        TParser parser(config.GetParser(), cookiesToCommit, unprocessedQueue, processorStats);

        NPQ::TReader reader(
                TDuration::MilliSeconds(config.GetReader().GetReadTimeoutMs()),
                consumer,
                cookiesToCommit,
                [&parser] (auto&& readResult) {
                    parser.Parse(std::move(readResult));
                },
                NLog::GetLog(logbrokerConfig.GetPqlibLogName())
        );
        reader.SetOnFinishHandler([&stopSignalWaiter] () {
            stopSignalWaiter.Signal();
        });

        log->info("Start");
        for (const auto& processor : processors) {
            processor->Start();
        }
        reader.Start();

        stopSignalWaiter.Wait();

        log->info("Stop");
    } catch (const yexception& e) {
        LogException(e.what());
        return 1;
    } catch (const std::exception& e) {
        LogException(e.what());
        return 1;
    } catch (...) {
        NLog::LogException("Unknown exception");
        return 1;
    }

    Cerr << "Shutting down YT library" << Endl;
    NYT::Shutdown();

    Cerr << "Exiting" << Endl;
    return 0;
}
