#include "parser.h"

#include <crypta/lib/native/time/shifted_clock.h>

#include <yt/yt/core/concurrency/scheduler.h>

#include <util/generic/algorithm.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>
#include <util/string/cast.h>
#include <util/string/split.h>

using namespace NCrypta::NExtFp;
using namespace NCrypta::NExtFp::NDelayLine;

TParser::TParser(const TParserConfig& config, NPQ::TCookieQueue& cookiesToCommit, TUnprocessedQueue<TVector<TParsedLine>>& unprocessedQueue, TStats& stats)
        : Sampler(config.GetSampler().GetDenominator(), config.GetSampler().GetRest())
        , ArtificialDelay(TDuration::Seconds(config.GetArtificialDelaySec()))
        , CookiesToCommit(cookiesToCommit)
        , UnprocessedQueue(unprocessedQueue)
        , Log(NLog::GetLog("parser"))
        , Stats(stats)
{
}

void TParser::Parse(NCrypta::NPQ::TConsumer::TReadResult&& readResult) {
    TLogParser::TParsedLines events;

    for (const auto& item : readResult.Data) {
        for (const auto& line : StringSplitter(item).Split('\n').SkipEmpty()) {
            try {
                const auto& event = TLogParser::ParseLine(line);
                if (!Sampler.Passes(event.GetYuid())) {
                    continue;
                }
                events.emplace_back(std::move(event));
            } catch (const yexception& e) {
                Stats.Count->Add("parser.errors.bad_event");
                Log->error("Cannot parse json: {}, {}", e.what(), line.Token());
            } catch (...) {
                Stats.Count->Add("parser.errors.bad_event");
                Log->error("Unknown exception: {}", line.Token());
            }
        }
    }

    if (events.size() > 0) {
       const auto maxEventTime = TInstant::Seconds(
               MaxElementBy(events, [](const auto& event){ return event.GetUnixtime(); })->GetUnixtime()
       );

        TDuration timeToSleep = ArtificialDelay - (TShiftedClock::Now() - maxEventTime);
        Sleep(timeToSleep);

        TUnprocessed<TVector<TParsedLine>> unprocessed(std::move(events));
        auto future = unprocessed.GetFuture();

        UnprocessedQueue.Enqueue(std::move(unprocessed));

        readResult.ClearData();

        NYT::NConcurrency::WaitFor(future).ThrowOnError();
    }

    CookiesToCommit.Enqueue(readResult.EpochCookie);
}
