#include "processor.h"

#include <crypta/lib/native/ext_fp/constants.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/time/sleeper.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/proto/ext_fp/fp_event.pb.h>

#include <util/stream/str.h>

using namespace NCrypta;
using namespace NCrypta::NExtFp;
using namespace NCrypta::NExtFp::NDelayLine;

TProcessor::TProcessor(
        TUnprocessedQueue<TVector<TEvent>>& unprocessedQueue,
        const TProcessorConfig& config,
        NPQ::TProducer& extFpEventLogProducer,
        TStats& stats
)
    : TThreaded("TProcessor")
    , UnprocessedQueue(unprocessedQueue)
    , Config(config)
    , MaxBatchingTime(TDuration::MilliSeconds(Config.GetMaxBatchingTimeMs()))
    , ExtFpEventLogProducer(extFpEventLogProducer)
    , Log(NLog::GetLog("processor"))
    , Stats(stats)
{
}

TProcessor::~TProcessor() {
    StopAndJoin();
}

void TProcessor::Run() {
    TBatch batch;
    TInstant nextDequeueTime;

    while (IsRunning()) {
        Sleep(nextDequeueTime - TInstant::Now());
        nextDequeueTime = MaxBatchingTime.ToDeadLine();

        batch.clear();
        const auto& eventCount = DequeueEvents(batch);
        Stats.Percentile->Add("event.count", eventCount);

        if (batch.empty()) {
            continue;
        }

        Stats.Percentile->Add("batch.size", batch.size());

        try {
            EnqueueBatch(batch);
        } catch (const yexception& e) {
            Stats.Count->Add("matching.errors");
            Log->error("Matching Error: {}", e.what());
        }

        RequestSuccess(batch);
    }
}

void TProcessor::EnqueueBatch(TBatch& batch) {
    for (auto& unprocessed : batch) {
        for (auto event : unprocessed.GetValue()) {
            const auto& nowSec = TShiftedClock::Now().Seconds();

            Stats.Percentile->Add("events.latency", nowSec - event.GetUnixtime());

            event.SetCurrentTimestamp(nowSec);
            ExtFpEventLogProducer.TryEnqueue(NCrypta::NProtoSerializer::ToJson(event));
        }
    }
}

ui32 TProcessor::DequeueEvents(TBatch& batch) {
    TUnprocessedEvents unprocessed = TVector<TEvent>();
    size_t eventCount = 0;

    while (eventCount <= Config.GetMaxEventsPerBatch() && UnprocessedQueue.Dequeue(&unprocessed)) {
        eventCount += unprocessed.GetValue().size();
        batch.emplace_back(std::move(unprocessed));
    }

    return eventCount;
}

void TProcessor::RequestSuccess(TBatch& batch) {
    for (auto& unprocessed : batch) {
        unprocessed.SetProcessed();
    }
}
