#include "beeline_matcher.h"

#include <crypta/ext_fp/matcher/lib/matchers/beeline_matcher/beeline_response_parser.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/https.h>

#include <util/generic/guid.h>
#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/string/split.h>

using namespace NCrypta::NExtFp::NMatcher;

TBeelineMatcher::TBeelineMatcher(const TBeelineMatcherConfig& config, const TNehSyncMultiClient& client, TStats& stats, NLog::TLogPtr log)
    : Config(config)
    , Client(client)
    , Stats(stats)
    , Log(log)
{
}

TConnection TBeelineMatcher::MakeConnection(const TFpEvent& event) {
    return {
        .Ip = event.GetIp(),
        .Port = event.GetPort(),
        .Timestamp = event.GetUnixtime(),
    };
}

void TBeelineMatcher::AddConnection(const TFpEvent& event) {
    Stats.Count->Add("events.incoming.beeline.count");
    Connections.emplace_back(MakeConnection(event));
}

TMatches TBeelineMatcher::GetMatches() {
    if (Connections.size() == 0) {
        return TMatches();
    }
    const auto& requestId = CreateGuidAsString();

    NNeh::TMessage message(GetApiUrl(), "");

    THttpHeaders headers;
    headers.AddHeader("Authorization", "api-key " + Config.GetToken());
    TStringStream serializedHeaders;
    headers.OutTo(&serializedHeaders);

    const auto& requestBody = GetRequestBody(Connections);
    Log->info("Beeline request {} body:\n{}", requestId, requestBody);

    Y_ENSURE(NNeh::NHttp::MakeFullRequest(message, serializedHeaders.Str(), requestBody, "application/json"), "Failed to build request to Beeline API");

    Stats.Count->Add("api.calls.beeline.count");
    const auto& resp = MakeRequest(Client, message, TDuration::MilliSeconds(Config.GetApiCallTimeoutMs()), "Beeline", requestId, Log);

    return NBeelineResponseParser::Parse(resp->Data);
}

TString TBeelineMatcher::GetApiUrl() const {
    return (Config.GetUseAuthentication() ? "posts://" : "post://") + Config.GetApiUrl();
}

TString TBeelineMatcher::GetRequestBody(const TVector<TConnection>& connections) {
    NJsonWriter::TBuf body;

    body.BeginList();

    for (const auto& connection : connections) {
        body.BeginObject()
            .WriteKey("ip").WriteString(connection.Ip)
            .WriteKey("port").WriteInt(connection.Port)
            .WriteKey("unixtime").WriteULongLong(connection.Timestamp)
            .EndObject();
    }

    body.EndList();

    return body.Str();
}
