#include "er_telecom_matcher.h"

#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/https.h>
#include <util/generic/guid.h>
#include <util/string/builder.h>
#include <util/string/split.h>

using namespace NCrypta::NExtFp::NMatcher;

namespace {
    TMatchResult::EStatus GetStatus(int status) {
        if (status < static_cast<int>(TMatchResult::EStatus::Found) || status > static_cast<int>(TMatchResult::EStatus::Error)) {
            return TMatchResult::EStatus::Unknown;
        }

        return static_cast<TMatchResult::EStatus>(status);
    }
}

TErTelecomMatcher::TErTelecomMatcher(const TErTelecomMatcherConfig& config, const TNehSyncMultiClient& client, TStats& stats, NLog::TLogPtr log)
    : Config(config)
    , Client(client)
    , Stats(stats)
    , Log(log)
{
}

TConnection TErTelecomMatcher::MakeConnection(const TFpEvent& event) {
    return {.Ip = event.GetIp(), .Port = event.GetPort(), .Timestamp = event.GetUnixtime(), .Domain = ""};
}

void TErTelecomMatcher::AddConnection(const TFpEvent& event) {
    auto connection = MakeConnection(event);

    Stats.Count->Add("events.incoming.ertelecom.count");
    Request += TStringBuilder() << connection.Ip << '\t'
                                << connection.Port << '\t'
                                << connection.Timestamp << '\n';
}

TMatches TErTelecomMatcher::GetMatches() {
    if (Request.length() == 0) {
        return TMatches();
    }
    const auto& requestId = CreateGuidAsString();
    Log->info("Er-Telecom request {} body:\n{}", requestId, Request);

    NNeh::TMessage message(GetApiUrl(), "");
    Y_ENSURE(NNeh::NHttp::MakeFullRequest(message, "", Request, "text/plain"), "Failed to build request to ER-Telecom API");

    Stats.Count->Add("api.calls.ertelecom.count");
    const auto& resp = MakeRequest(Client, message, TDuration::MilliSeconds(Config.GetApiCallTimeoutMs()), "Er-Telecom", requestId, Log);

    return ParseResponse(resp->Data);
}

TString TErTelecomMatcher::GetApiUrl() const {
    if (Config.GetUseAuthentication()) {
        return "posts://cert=" + Config.GetCrtFilePath() + ";key=" + Config.GetKeyFilePath() + "@" + Config.GetApiUrl();
    } else {
        return "post://" + Config.GetApiUrl();
    }
}

TMatches TErTelecomMatcher::ParseResponse(const TString& response) {
    TStringInput stringInput(response);

    TString line;
    TConnection connection;
    int status = 0;
    TString extId;

    TMatches matches;

    while (stringInput.ReadLine(line)) {
        Split(TStringBuf(line), '\t', status, connection.Ip, connection.Port, connection.Timestamp, extId);
        matches[connection] = {.Status = GetStatus(status), .ExtId = extId};
    }

    return matches;
}
