#include "intentai_matcher.h"

#include <crypta/ext_fp/matcher/lib/matchers/intentai_matcher/intentai_response_parser.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/https.h>

#include <util/generic/guid.h>
#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/string/split.h>

using namespace NCrypta::NExtFp::NMatcher;

TIntentaiMatcher::TIntentaiMatcher(const TIntentaiMatcherConfig& config, const TNehSyncMultiClient& client, TStats& stats, NLog::TLogPtr log)
    : Config(config)
    , Client(client)
    , Stats(stats)
    , Log(log)
{
}

TConnection TIntentaiMatcher::MakeConnection(const TFpEvent& event) {
    return {
        .Ip = event.GetIp(),
        .Port = event.GetPort(),
        .Timestamp = event.GetUnixtime(),
    };
}

void TIntentaiMatcher::AddConnection(const TFpEvent& event) {
    Stats.Count->Add("events.incoming.intentai.count");
    Connections.emplace_back(MakeConnection(event));
}

TMatches TIntentaiMatcher::GetMatches() {
    if (Connections.size() == 0) {
        return TMatches();
    }
    const auto& requestId = CreateGuidAsString();

    NNeh::TMessage message(GetApiUrl(), "");

    THttpHeaders headers;
    headers.AddHeader("Intent-Api-Key", Config.GetToken());
    TStringStream serializedHeaders;
    headers.OutTo(&serializedHeaders);

    const auto& requestBody = GetRequestBody(Connections);
    Log->info("Intentai request {} body:\n{}", requestId, requestBody);

    Y_ENSURE(NNeh::NHttp::MakeFullRequest(message, serializedHeaders.Str(), requestBody, "application/json"), "Failed to build request to Intentai API");

    Stats.Count->Add("api.calls.intentai.count");
    const auto& resp = MakeRequest(Client, message, TDuration::MilliSeconds(Config.GetApiCallTimeoutMs()), "Intentai", requestId, Log);

    return NIntentaiResponseParser::Parse(resp->Data);
}

TString TIntentaiMatcher::GetApiUrl() const {
    return (Config.GetUseAuthentication() ? "posts://" : "post://") + Config.GetApiUrl();
}

TString TIntentaiMatcher::GetRequestBody(const TVector<TConnection>& connections) {
    NJsonWriter::TBuf body;

    body.BeginList();

    for (const auto& connection : connections) {
        body.BeginObject()
            .WriteKey("ip").WriteString(connection.Ip)
            .WriteKey("port").WriteInt(connection.Port)
            .WriteKey("timestamp").WriteULongLong(connection.Timestamp)
            .EndObject();
    }

    body.EndList();

    return body.Str();
}
